<?php
// Növeljük a lehetséges futási időt és memóriát a képfeldolgozáshoz
set_time_limit(60); // 60 másodperc
ini_set('memory_limit', '256M'); // 256 MB memória

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: https://www.homeinfo.hu");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// --- Adatbázis kapcsolódási adatok ---
// !!! FONTOS: Éles környezetben ezeket ne így tárold! Használj környezeti változókat vagy config fájlt!
define('DB_HOST', 'localhost'); // Vagy az adatbázis szerver címe
define('DB_NAME', 'dasl1_db_inspiracio'); // Cseréld le az adatbázis nevére
define('DB_USER', 'dasl1_usr_inspir'); // Cseréld le a felhasználónévre
define('DB_PASS', 'p-^u;.TdrB*5'); // Cseréld le a jelszóra
define('UPLOAD_DIR', __DIR__ . '/uploads/'); // Feltöltési könyvtár útvonala
define('UPLOAD_URL', 'uploads/'); // URL a képek eléréséhez a frontendről
define('MAX_IMAGE_WIDTH', 800); // Maximális képszélesség pixelben
define('JPEG_QUALITY', 80); // JPEG tömörítési minőség (0-100)
// --------------------------------------

try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION, // Hibák esetén kivételt dob
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC, // Asszociatív tömbként adja vissza az eredményt
            PDO::ATTR_EMULATE_PREPARES => false, // Valódi prepared statement-eket használ
        ]
    );
} catch (PDOException $e) {
    http_response_code(500); // Internal Server Error
    echo json_encode(['error' => 'Adatbázis kapcsolódási hiba: ' . $e->getMessage()]);
    exit();
}



// --- Segédfüggvény a JSON input olvasásához ---
function getJsonInput()
{
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Érvénytelen JSON formátum.");
    }
    return $data;
}
// Képfeldolgozó és mentő függvény
function processUploadedImage(array $fileInfo, string $uploadDir): ?string
{
    // Alapvető hibák ellenőrzése
    if (!isset($fileInfo['error']) || is_array($fileInfo['error']) || $fileInfo['error'] !== UPLOAD_ERR_OK) {
        throw new RuntimeException('Érvénytelen feltöltési paraméterek vagy hiba a feltöltés során. Kód: ' . ($fileInfo['error'] ?? 'N/A'));
    }
    if ($fileInfo['size'] == 0) {
        throw new RuntimeException('A feltöltött fájl üres.');
    }
    // Méretkorlát (pl. 5MB)
    if ($fileInfo['size'] > 5 * 1024 * 1024) {
        throw new RuntimeException('A fájl mérete túl nagy (max 5MB).');
    }

    // MIME típus ellenőrzése (biztonságosabb módszer)
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->file($fileInfo['tmp_name']);
    $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!in_array($mimeType, $allowedMimeTypes)) {
        throw new RuntimeException('Nem engedélyezett fájltípus. Csak JPG, PNG, GIF, WEBP tölthető fel.');
    }

    // GD kiterjesztés ellenőrzése
    if (!extension_loaded('gd')) {
        throw new RuntimeException('A GD kiterjesztés nincs engedélyezve a szerveren a képfeldolgozáshoz.');
    }

    // Kép betöltése GD-vel
    $image = null;
    switch ($mimeType) {
        case 'image/jpeg':
            $image = @imagecreatefromjpeg($fileInfo['tmp_name']);
            break;
        case 'image/png':
            $image = @imagecreatefrompng($fileInfo['tmp_name']);
            break;
        case 'image/gif':
            $image = @imagecreatefromgif($fileInfo['tmp_name']);
            break;
        case 'image/webp':
            $image = @imagecreatefromwebp($fileInfo['tmp_name']);
            break;
    }
    if (!$image) {
        throw new RuntimeException('Nem sikerült a képfájl betöltése (lehet, hogy sérült).');
    }

    // Eredeti méretek
    $originalWidth = imagesx($image);
    $originalHeight = imagesy($image);

    // Új méretek kiszámítása (arányos kicsinyítés)
    $newWidth = $originalWidth;
    $newHeight = $originalHeight;
    if ($originalWidth > MAX_IMAGE_WIDTH) {
        $ratio = MAX_IMAGE_WIDTH / $originalWidth;
        $newWidth = MAX_IMAGE_WIDTH;
        $newHeight = (int)($originalHeight * $ratio);
    }

    // Új (átméretezett) kép létrehozása
    $resizedImage = imagecreatetruecolor($newWidth, $newHeight);

    // Átlátszóság kezelése PNG és GIF esetén
    if ($mimeType == 'image/png' || $mimeType == 'image/gif') {
        imagealphablending($resizedImage, false); // Kikapcsoljuk az alapértelmezett blendinget
        imagesavealpha($resizedImage, true);      // Engedélyezzük az alpha csatorna mentését
        $transparentColor = imagecolorallocatealpha($resizedImage, 255, 255, 255, 127); // Teljesen átlátszó szín
        imagefill($resizedImage, 0, 0, $transparentColor); // Kitöltjük az új képet átlátszó színnel
        imagecolortransparent($resizedImage, $transparentColor); // GIF-hez is beállítjuk
    } else {
        // JPEG-hez fehér háttér (ha esetleg átlátszó PNG-ből konvertálnánk)
        $white = imagecolorallocate($resizedImage, 255, 255, 255);
        imagefilledrectangle($resizedImage, 0, 0, $newWidth, $newHeight, $white);
    }


    // Kép átméretezése és másolása
    imagecopyresampled(
        $resizedImage,
        $image,
        0,
        0,
        0,
        0,
        $newWidth,
        $newHeight,
        $originalWidth,
        $originalHeight
    );

    // Célfájl nevének generálása (egyedi név + eredeti kiterjesztés helyett fixen jpg)
    $extension = 'jpg'; // Mindent JPG-be mentünk az egyszerűség kedvéért
    $safeFilename = preg_replace('/[^a-zA-Z0-9_-]/', '_', pathinfo($fileInfo['name'], PATHINFO_FILENAME)); // Eredeti név "tisztítása"
    $newFilename = uniqid($safeFilename . '_', true) . '.' . $extension;
    $destinationPath = rtrim($uploadDir, '/') . '/' . $newFilename;

    // Kép mentése JPEG formátumban
    if (!imagejpeg($resizedImage, $destinationPath, JPEG_QUALITY)) {
        imagedestroy($image);
        imagedestroy($resizedImage);
        throw new RuntimeException('Nem sikerült az optimalizált kép mentése.');
    }

    // Memória felszabadítása
    imagedestroy($image);
    imagedestroy($resizedImage);

    // Visszaadjuk az új fájlnevet
    return $newFilename;
}

// Fájl törlése (ha létezik)
function deleteImageFile(?string $filename, string $uploadDir): bool
{
    if (empty($filename)) {
        return false;
    }
    $filePath = rtrim($uploadDir, '/') . '/' . $filename;
    if (file_exists($filePath) && is_writable($filePath)) {
        return unlink($filePath);
    }
    return false;
}

/**
 * Megtisztítja a kérdés szövegét a nem engedélyezett HTML tagektől.
 * Csak az <a> taget engedélyezi 'href' és 'target' attribútumokkal.
 *
 * @param string|null $text A bejövő szöveg.
 * @return string A megtisztított szöveg.
 */
function sanitizeQuestionText(?string $text): string
{
    if ($text === null) {
        return '';
    }
    // strip_tags eltávolít minden HTML és PHP taget, kivéve a megadottakat.
    // A '<a>' tag mellé megadhatjuk az engedélyezett attribútumokat is.
    $allowed_tags = '<a>';
    return strip_tags($text, $allowed_tags);
}

// --- API Végpontok ---
$action = $_GET['action'] ?? null;
$method = $_SERVER['REQUEST_METHOD'];
try {
    // Ellenőrizzük az uploads mappa létezését és írhatóságát
    if (!is_dir(UPLOAD_DIR) || !is_writable(UPLOAD_DIR)) {
        throw new RuntimeException("A feltöltési könyvtár ('" . htmlspecialchars(UPLOAD_DIR) . "') nem létezik vagy nem írható a szerver által.");
    }
    switch ($action) {
        // === Kérdéscsoportok ===
        case 'get_groups':
            if ($method === 'GET') {
                $stmt = $pdo->query("SELECT id, group_name, release_date FROM quiz_question_groups ORDER BY release_date DESC, group_name ASC");
                $groups = $stmt->fetchAll();
                echo json_encode($groups);
            } else {
                http_response_code(405); // Method Not Allowed
                echo json_encode(['error' => 'Csak GET kérés engedélyezett.']);
            }
            break;

        case 'add_group':
            if ($method === 'POST') {
                $data = getJsonInput();
                $groupName = $data['group_name'] ?? null;
                $releaseDate = $data['release_date'] ?? null;

                if (empty($groupName) || empty($releaseDate)) {
                    http_response_code(400); // Bad Request
                    echo json_encode(['error' => 'Hiányzó adatok (group_name, release_date).']);
                    exit;
                }

                // Validálás (dátum formátum)
                if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $releaseDate)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Érvénytelen dátum formátum (YYYY-MM-DD szükséges).']);
                    exit;
                }

                $sql = "INSERT INTO quiz_question_groups (group_name, release_date) VALUES (:group_name, :release_date)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute(['group_name' => $groupName, 'release_date' => $releaseDate]);
                $newGroupId = $pdo->lastInsertId();

                http_response_code(201); // Created
                echo json_encode(['id' => $newGroupId, 'group_name' => $groupName, 'release_date' => $releaseDate, 'message' => 'Csoport sikeresen hozzáadva.']);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett.']);
            }
            break;

        // === Kérdések ===
        case 'get_questions':
            if ($method === 'GET') {
                $groupId = $_GET['group_id'] ?? null;
                if (!$groupId) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó group_id paraméter.']);
                    exit;
                }
                $stmt = $pdo->prepare("SELECT id, question_text, question_image, question_type FROM quiz_questions WHERE question_group_id = :group_id ORDER BY id ASC");
                $stmt->execute(['group_id' => $groupId]);
                $questions = $stmt->fetchAll();
                // Kép URL hozzáadása, ha van képnév
                foreach ($questions as &$q) {
                    $q['question_image_url'] = $q['question_image'] ? UPLOAD_URL . $q['question_image'] : null;
                }
                unset($q);
                echo json_encode($questions);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak GET kérés engedélyezett.']);
            }
            break;

        // MÓDOSÍTOTT: Kérdés hozzáadása (FormData-t vár!)
        case 'add_question':
            if ($method === 'POST') {
                // Adatok olvasása $_POST-ból
                $groupId = filter_input(INPUT_POST, 'group_id', FILTER_VALIDATE_INT);
                //$questionText = filter_input(INPUT_POST, 'question_text', FILTER_SANITIZE_STRING);
                $questionText = sanitizeQuestionText($_POST['question_text'] ?? null);
                $questionType = filter_input(INPUT_POST, 'question_type', FILTER_SANITIZE_STRING);

                if (!$groupId || empty($questionText) || empty($questionType)) { // Type is legyen kötelező
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó adatok (group_id, question_text, question_type).']);
                    exit;
                }

                $newImageFilename = null;
                // Képfájl feldolgozása, ha van
                if (isset($_FILES['question_image_file']) && $_FILES['question_image_file']['error'] === UPLOAD_ERR_OK) {
                    try {
                        $newImageFilename = processUploadedImage($_FILES['question_image_file'], UPLOAD_DIR);
                    } catch (RuntimeException $e) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Képfeldolgozási hiba: ' . $e->getMessage()]);
                        exit;
                    }
                }

                // Beszúrás adatbázisba
                $sql = "INSERT INTO quiz_questions (question_group_id, question_text, question_image, question_type) VALUES (:group_id, :qtext, :qimage, :qtype)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'group_id' => $groupId,
                    'qtext' => $questionText,
                    'qimage' => $newImageFilename, // Az új, optimalizált kép neve
                    'qtype' => $questionType
                ]);
                $newQuestionId = $pdo->lastInsertId();

                http_response_code(201);
                echo json_encode([
                    'id' => $newQuestionId,
                    'question_group_id' => $groupId,
                    'question_text' => $questionText,
                    'question_image' => $newImageFilename, // Visszaküldjük a fájlnevet
                    'question_image_url' => $newImageFilename ? UPLOAD_URL . $newImageFilename : null, // És a teljes URL-t
                    'question_type' => $questionType,
                    'message' => 'Kérdés sikeresen hozzáadva.'
                ]);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett.']);
            }
            break;

        // ÚJ: Kérdés módosítása (FormData-t vár!)
        case 'update_question':
            if ($method === 'POST') { // POST-ot használunk FormData miatt
                // Adatok olvasása $_POST-ból
                $questionId = filter_input(INPUT_POST, 'question_id', FILTER_VALIDATE_INT);
                //$questionText = filter_input(INPUT_POST, 'question_text', FILTER_SANITIZE_STRING);
                $questionText = sanitizeQuestionText($_POST['question_text'] ?? null);
                $questionType = filter_input(INPUT_POST, 'question_type', FILTER_SANITIZE_STRING);
                $removeImage = filter_input(INPUT_POST, 'remove_image', FILTER_VALIDATE_BOOLEAN); // Flag a kép törléséhez

                if (!$questionId || empty($questionText) || empty($questionType)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó adatok (question_id, question_text, question_type).']);
                    exit;
                }

                // Aktuális kérdés adatainak lekérése (főleg a képnév miatt)
                $stmt = $pdo->prepare("SELECT question_image FROM quiz_questions WHERE id = :id");
                $stmt->execute(['id' => $questionId]);
                $currentQuestion = $stmt->fetch();

                if (!$currentQuestion) {
                    http_response_code(404);
                    echo json_encode(['error' => 'A megadott ID-vel nem található kérdés.']);
                    exit;
                }
                $currentImageFilename = $currentQuestion['question_image'];
                $newImageFilename = $currentImageFilename; // Alapból marad a régi

                // Van új kép feltöltve?
                if (isset($_FILES['question_image_file']) && $_FILES['question_image_file']['error'] === UPLOAD_ERR_OK) {
                    try {
                        // Régi kép törlése (ha volt) MIUTÁN az új feldolgozása sikeres volt
                        $uploadedFilename = processUploadedImage($_FILES['question_image_file'], UPLOAD_DIR);
                        if ($uploadedFilename) {
                            deleteImageFile($currentImageFilename, UPLOAD_DIR); // Régi törlése
                            $newImageFilename = $uploadedFilename; // Új kép nevének beállítása
                        }
                    } catch (RuntimeException $e) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Képfeldolgozási hiba: ' . $e->getMessage()]);
                        exit;
                    }
                }
                // Vagy törölni kell a képet? (csak ha nem töltöttek fel újat közben)
                elseif ($removeImage && $currentImageFilename) {
                    deleteImageFile($currentImageFilename, UPLOAD_DIR);
                    $newImageFilename = null; // Képnév nullázása a DB-ben
                }

                // Adatbázis frissítése
                $sql = "UPDATE quiz_questions SET question_text = :qtext, question_image = :qimage, question_type = :qtype WHERE id = :id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'id' => $questionId,
                    'qtext' => $questionText,
                    'qimage' => $newImageFilename,
                    'qtype' => $questionType
                ]);

                http_response_code(200);
                echo json_encode([
                    'id' => $questionId,
                    'question_text' => $questionText,
                    'question_image' => $newImageFilename,
                    'question_image_url' => $newImageFilename ? UPLOAD_URL . $newImageFilename : null,
                    'question_type' => $questionType,
                    'message' => 'Kérdés sikeresen módosítva.'
                ]);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett.']);
            }
            break;

        case 'delete_question':
            if ($method === 'POST') {
                $questionId = $_POST['question_id'] ?? null;
                if (!$questionId) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó question_id paraméter.']);
                    exit;
                }
                $stmt = $pdo->prepare("SELECT question_image FROM quiz_questions WHERE id = :id");
                $stmt->execute(['id' => $questionId]);
                $question = $stmt->fetch();
                if (!$question) {
                    http_response_code(404);
                    echo json_encode(['error' => 'A megadott ID-vel nem található kérdés.']);
                    exit;
                }
                $imageFilename = $question['question_image'];
                if ($imageFilename) {
                    deleteImageFile($imageFilename, UPLOAD_DIR);
                }
                $stmt = $pdo->prepare("DELETE FROM quiz_questions WHERE id = :id");
                $stmt->execute(['id' => $questionId]);
                http_response_code(200);
                echo json_encode(['success' => true, 'message' => 'Kérdés sikeresen töröltve.']);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett.']);
            }
            break;

        // === Válaszok ===
        case 'get_answers':
            if ($method === 'GET') {
                $questionId = $_GET['question_id'] ?? null;
                if (!$questionId) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó question_id paraméter.']);
                    exit;
                }
                $stmt = $pdo->prepare("SELECT id, answer_text, is_correct FROM quiz_answers WHERE question_id = :question_id ORDER BY id ASC");
                $stmt->execute(['question_id' => $questionId]);
                $answers = $stmt->fetchAll();
                // A 'is_correct' mezőt boolean-ként küldjük a frontendnek
                foreach ($answers as &$answer) {
                    $answer['is_correct'] = (bool)$answer['is_correct'];
                }
                unset($answer); // Referencia törlése

                echo json_encode($answers);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak GET kérés engedélyezett.']);
            }
            break;

        case 'add_answer':
            if ($method === 'POST') {
                $data = getJsonInput();
                $questionId = $data['question_id'] ?? null;
                $answerText = $data['answer_text'] ?? null;
                // A frontend boolean-t küld, a DB tinyint(1)-et vár (0 vagy 1)
                $isCorrect = isset($data['is_correct']) ? ($data['is_correct'] ? 1 : 0) : null;

                if (!$questionId || empty($answerText) || $isCorrect === null) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Hiányzó adatok (question_id, answer_text, is_correct).']);
                    exit;
                }

                $sql = "INSERT INTO quiz_answers (question_id, answer_text, is_correct) VALUES (:question_id, :atext, :is_correct)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'question_id' => $questionId,
                    'atext' => $answerText,
                    'is_correct' => $isCorrect
                ]);
                $newAnswerId = $pdo->lastInsertId();

                http_response_code(201);
                echo json_encode([
                    'id' => $newAnswerId,
                    'question_id' => $questionId,
                    'answer_text' => $answerText,
                    'is_correct' => (bool)$isCorrect, // Vissza boolean-ként
                    'message' => 'Válasz sikeresen hozzáadva.'
                ]);
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett.']);
            }
            break;
        // === ÚJ: Válasz módosítása ===
        case 'update_answer':
            if ($method === 'POST') { // Vagy lehetne PUT is, de maradjunk POST-nál az egyszerűség kedvéért
                $data = getJsonInput();
                $answerId = $data['id'] ?? null; // Kell az ID a módosításhoz!
                $answerText = $data['answer_text'] ?? null;
                $isCorrect = isset($data['is_correct']) ? ($data['is_correct'] ? 1 : 0) : null;

                if (!$answerId || empty($answerText) || $isCorrect === null) {
                    http_response_code(400); // Bad Request
                    echo json_encode(['error' => 'Hiányzó vagy érvénytelen adatok (id, answer_text, is_correct).']);
                    exit;
                }

                // Ellenőrizzük, létezik-e a válasz
                $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM quiz_answers WHERE id = :id");
                $checkStmt->execute(['id' => $answerId]);
                if ($checkStmt->fetchColumn() == 0) {
                    http_response_code(404); // Not Found
                    echo json_encode(['error' => 'A megadott ID-vel nem található válasz.']);
                    exit;
                }


                $sql = "UPDATE quiz_answers SET answer_text = :atext, is_correct = :is_correct WHERE id = :id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'id' => $answerId,
                    'atext' => $answerText,
                    'is_correct' => $isCorrect
                ]);

                // Ellenőrizzük, hogy történt-e módosítás (rowCount > 0)
                if ($stmt->rowCount() > 0) {
                    http_response_code(200); // OK
                    echo json_encode([
                        'id' => $answerId,
                        'answer_text' => $answerText,
                        'is_correct' => (bool)$isCorrect,
                        'message' => 'Válasz sikeresen módosítva.'
                    ]);
                } else {
                    // Nem történt módosítás (lehet, hogy ugyanazokat az adatokat küldték)
                    // Vagy visszaküldhetjük a meglévő adatokat, vagy csak egy sima OK üzenetet
                    http_response_code(200); // Vagy 304 Not Modified, de egyszerűbb a 200
                    echo json_encode([
                        'id' => $answerId,
                        'answer_text' => $answerText,
                        'is_correct' => (bool)$isCorrect,
                        'message' => 'A válasz adatai nem változtak.'
                    ]);
                    // Alternatíva:
                    // http_response_code(500);
                    // echo json_encode(['error' => 'A válasz módosítása sikertelen.']);
                }
            } else {
                http_response_code(405);
                echo json_encode(['error' => 'Csak POST kérés engedélyezett ehhez a végponthoz.']);
            }
            break;

        default:
            http_response_code(404); // Not Found
            echo json_encode(['error' => 'Érvénytelen API végpont vagy action paraméter.']);
            break;
    }
} catch (PDOException $e) {
    http_response_code(500);
    error_log("PDO Hiba: " . $e->getMessage() . " SQL: " . ($stmt ? $stmt->queryString : 'N/A'));
    echo json_encode(['error' => 'Adatbázis hiba történt a művelet során.']);
} catch (RuntimeException $e) { // Elkapjuk a képfeldolgozási és egyéb hibákat
    http_response_code(500); // Vagy 400, kontextustól függően
    error_log("Runtime Hiba: " . $e->getMessage());
    echo json_encode(['error' => $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(400);
    error_log("Általános Hiba: " . $e->getMessage());
    echo json_encode(['error' => 'Hiba történt: ' . $e->getMessage()]);
}
