document.addEventListener('DOMContentLoaded', () => {
    // API URL
    const API_URL = 'https://www.homeinfo.hu/vip/admin/api.php';
    const UPLOAD_URL_BASE = 'uploads/'; // Ezt használjuk az URL-ek összeállításához

    // === DOM Elemek ===
    const loader = document.getElementById('loader');
    const messageArea = document.getElementById('message-area');

    // Nézetek
    const groupsView = document.getElementById('groups-view');
    const questionsView = document.getElementById('questions-view');
    const answersView = document.getElementById('answers-view');

    // Listák
    const groupsList = document.getElementById('groups-list');
    const questionsList = document.getElementById('questions-list'); // Event delegation-höz kell
    const answersList = document.getElementById('answers-list');

    // Formok és gombok
    const addGroupForm = document.getElementById('add-group-form');
    
    // === Kérdés Form Elemek ===
    const questionFormSection = document.getElementById('question-form-section');
    const addQuestionForm = document.getElementById('add-question-form');
    const questionFormTitle = document.getElementById('question-form-title');
    const questionSubmitButton = document.getElementById('question-submit-button');
    const cancelQuestionEditButton = document.getElementById('cancel-question-edit-button');
    const questionGroupIdInput = document.getElementById('question_group_id'); // Marad a csoport ID-hez
    const editingQuestionIdInput = document.getElementById('editing_question_id'); // Szerkesztett kérdés ID
    const questionTextInput = document.getElementById('question_text');
    const questionTypeSelect = document.getElementById('question_type');
    const questionImageFileInput = document.getElementById('question_image_file');
    const imagePreviewContainer = document.getElementById('image-preview-container');
    const imagePreview = document.getElementById('image-preview');
    const removeImageButton = document.getElementById('remove-image-button');
    const removeImageFlagInput = document.getElementById('remove_image_flag');
    const currentImageUrlInput = document.getElementById('current_image_url'); // Tárolja az aktuális kép URL-t szerkesztéskor

    // === Válasz Form Elemek (változatlan nevek) ===
    const answerFormSection = document.getElementById('answer-form-section');
    const addAnswerForm = document.getElementById('add-answer-form');
    const answerFormTitle = document.getElementById('answer-form-title');
    const answerSubmitButton = document.getElementById('answer-submit-button');
    const cancelEditButton = document.getElementById('cancel-edit-button'); // Válasz form "Mégse"
    const answerQuestionIdInput = document.getElementById('answer_question_id');
    const editingAnswerIdInput = document.getElementById('editing_answer_id');

    // Vissza gombok
    const backToGroupsBtn = document.getElementById('back-to-groups');
    const backToQuestionsBtn = document.getElementById('back-to-questions');

    // Kijelző elemek
    const selectedGroupNameEl = document.getElementById('selected-group-name');
    const selectedQuestionTextEl = document.getElementById('selected-question-text');

    // === Állapot ===
    let currentGroupId = null;
    let currentGroupName = '';
    let currentQuestionId = null; // Válaszokhoz használt
    let currentQuestionText = ''; // Válaszokhoz használt
    let editingQuestionId = null; // Kérdés szerkesztés állapot
    let editingAnswerId = null; // Válasz szerkesztés állapot

    // --- Segédfüggvények ---
    // showLoader, hideLoader, showMessage, handleApiError, fetchData (változatlanok)
    const showLoader = () => loader.classList.remove('hidden');
    const hideLoader = () => loader.classList.add('hidden');

    const showMessage = (message, isError = false) => {
        messageArea.textContent = message;
        messageArea.className = `mb-4 text-center p-2 rounded ${isError ? 'bg-red-100 text-red-700' : 'bg-green-100 text-green-700'}`;
        setTimeout(() => {
             // Csak akkor töröljük, ha még mindig ez az üzenet van ott
             if (messageArea.textContent === message) {
                 messageArea.textContent = '';
                 messageArea.className = 'mb-4 text-center'; // Stílusok eltávolítása
             }
        }, 5000);
    };

    const handleApiError = (error) => {
        console.error('API Hiba:', error);
        let displayMessage = 'Ismeretlen API hiba történt.';
        // Próbáljuk meg az error objektum message property-jét használni, ha van
        if (error && error.message) {
            // Ne jelenítsük meg a "Failed to fetch" üzenetet direktben a usernek
            if (error.message.toLowerCase().includes('failed to fetch')) {
                 displayMessage = 'Hálózati hiba vagy a szerver nem elérhető.';
            } else {
                 displayMessage = error.message;
            }
        }
        showMessage(displayMessage, true);
    }

    // MÓDOSÍTOTT: fetchData (már nem mindig JSON a válasz, de a hibakezelés hasonló)
    // A FormData küldésénél a Content-Type-ot NEM adjuk meg headerben!
    const fetchData = async (url, options = {}) => {
        showLoader();
        try {
            // Ha FormData van a body-ban, a böngésző beállítja a Content-Type-ot
            // így töröljük, ha manuálisan be volt állítva JSON-ra
            if (options.body instanceof FormData && options.headers && options.headers['Content-Type']) {
                delete options.headers['Content-Type'];
            } else if (!(options.body instanceof FormData) && !options.headers?.['Content-Type']) {
                 // Ha nem FormData és nincs header, feltételezzük a JSON-t (pl. válaszoknál)
                 options.headers = { ...options.headers, 'Content-Type': 'application/json' };
            }


            const response = await fetch(url, options);
            const responseText = await response.text(); // Mindig szövegként olvassuk először
            let data;
            try {
                data = JSON.parse(responseText); // Próbáljuk JSON-ként parse-olni
            } catch (e) {
                 // Ha a parse nem sikerült, de a válasz OK volt
                 if (response.ok) {
                     console.warn("Nem JSON válasz érkezett, de a státusz OK:", responseText);
                     // Lehet, hogy csak egy sima szöveges üzenet jött, vagy üres volt
                     data = { success: true, message: responseText || "Sikeres művelet." };
                 } else {
                    // Ha a parse nem sikerült ÉS a státusz sem OK
                    console.error("JSON Parse Hiba:", responseText);
                    throw new Error(`Érvénytelen válasz a szerverről (Status: ${response.status}). Tartalom: ${responseText.substring(0, 100)}...`);
                 }
            }

            if (!response.ok) {
                const errorMsg = data?.error || `HTTP hiba: ${response.status} ${response.statusText}`;
                throw new Error(errorMsg);
            }
            return data; // Sikeres válasz (JSON objektum)
        } catch (error) {
            handleApiError(error);
            return null;
        } finally {
            hideLoader();
        }
    };

     // --- Nézetváltás ---
     const showView = (viewToShow) => {
        [groupsView, questionsView, answersView].forEach(view => {
            view.classList.toggle('hidden-view', view.id !== viewToShow);
            view.classList.toggle('visible-view', view.id === viewToShow);
        });
        // Formok resetelése nézetváltáskor
        if (viewToShow !== 'questions-view' && editingQuestionId !== null) {
           resetQuestionForm();
        }
        if (viewToShow !== 'answers-view' && editingAnswerId !== null) {
           resetAnswerForm();
        }
        if (document.activeElement?.form?.id !== 'add-question-form' && document.activeElement?.form?.id !== 'add-answer-form') {
           window.scrollTo(0, 0);
        }
    };

    // --- Renderelők ---

    // renderGroups, renderQuestions (változatlanok)
    const renderGroups = (groups) => {
        groupsList.innerHTML = ''; // Lista ürítése
        if (!groups || groups.length === 0) {
            groupsList.innerHTML = '<p class="text-gray-500">Nincsenek kérdéscsoportok.</p>';
            return;
        }
        groups.forEach(group => {
            const div = document.createElement('div');
            div.className = 'p-3 border rounded hover:bg-gray-50 cursor-pointer flex justify-between items-center';
            div.innerHTML = `
                <span>
                    <strong class="text-blue-700">${group.group_name}</strong>
                    <span class="text-sm text-gray-500 ml-2">(Megjelenik: ${group.release_date || 'Nincs dátum'})</span>
                </span>
                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 text-gray-400" viewBox="0 0 20 20" fill="currentColor">
                  <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd" />
                </svg>
            `;
            div.addEventListener('click', () => loadQuestions(group.id, group.group_name));
            groupsList.appendChild(div);
        });
    };

    // MÓDOSÍTOTT: renderQuestions (szerkesztés gombbal és kép megjelenítéssel)
    const renderQuestions = (questions) => {
        questionsList.innerHTML = '';
        if (!questions || questions.length === 0) {
            questionsList.innerHTML = '<p class="text-gray-500">Nincsenek kérdések ebben a csoportban.</p>';
            return;
        }
        questions.forEach(question => {
            const div = document.createElement('div');
            div.className = 'p-3 border rounded bg-white flex justify-between items-start question-item'; // items-start a több soros szöveghez
            div.dataset.questionId = question.id; // ID tárolása szerkesztéshez
            // A teljes question objektum tárolása a datasetben (stringként)
            div.dataset.questionData = JSON.stringify(question);

            div.innerHTML = `
                <div class="flex-grow mr-4">
                    <div class="mb-1"> <!-- Kérdés szövege és típus -->
                       <span class="question-text-display">${question.question_text}</span>
                       ${question.question_type ? `<span class="ml-2 text-xs bg-blue-100 text-blue-800 px-1.5 py-0.5 rounded">${question.question_type}</span>` : ''}
                    </div>
                     ${question.question_image_url ?
                        `<img src="${question.question_image_url}?t=${new Date().getTime()}" alt="Kérdés képe" class="image-preview my-1" loading="lazy">` :
                        '' /* Nincs kép, nincs img tag */
                     }
                </div>
                <div class="flex-shrink-0 flex flex-col items-end space-y-1"> <!-- Gombok egymás alatt -->
                    <button class="edit-question-btn text-blue-600 hover:text-blue-800 edit-btn border border-blue-300 rounded bg-blue-50 hover:bg-blue-100 px-2 py-1 text-xs"
                            aria-label="Kérdés szerkesztése">
                        Szerkesztés
                    </button>
                    <button class="view-answers-btn text-purple-600 hover:text-purple-800 edit-btn border border-purple-300 rounded bg-purple-50 hover:bg-purple-100 px-2 py-1 text-xs"
                            aria-label="Válaszok megtekintése/szerkesztése">
                        Válaszok
                    </button>
                    <button class="delete-question-btn text-red-600 hover:text-red-800 edit-btn border border-red-300 rounded bg-red-50 hover:bg-red-100 px-2 py-1 text-xs"
                            aria-label="Kérdés törlése">
                        Törlés
                    </button>
                </div>
            `;
            questionsList.appendChild(div);
        });
    };

    // MÓDOSÍTOTT RENDERER
    const renderAnswers = (answers) => {
        answersList.innerHTML = '';
        if (!answers || answers.length === 0) {
            answersList.innerHTML = '<p class="text-gray-500">Nincsenek válaszok ehhez a kérdéshez.</p>';
            return;
        }
        answers.forEach(answer => {
            const div = document.createElement('div');
            const isCorrectClass = answer.is_correct ? 'border-l-4 border-green-500 correct-answer' : 'border-l-4 border-gray-300';
            // Data attribútumok hozzáadása a szerkesztéshez
            div.className = `p-3 border rounded bg-white ${isCorrectClass} flex justify-between items-center answer-item`;
            div.dataset.answerId = answer.id; // Fontos: ID tárolása
            div.dataset.answerText = answer.answer_text; // Szöveg tárolása
            div.dataset.isCorrect = answer.is_correct; // Helyesség tárolása (string "true"/"false")

            div.innerHTML = `
                <span class="flex-grow mr-2">${answer.answer_text}</span>
                <span class="flex-shrink-0"> <!-- Gombok ne törjenek -->
                    ${answer.is_correct ? '<span class="text-green-600 font-semibold mr-3">Helyes</span>' : '<span class="text-gray-400 font-semibold mr-3">Helytelen</span>'}
                    <button class="edit-answer-btn text-blue-600 hover:text-blue-800 edit-btn border border-blue-300 rounded bg-blue-50 hover:bg-blue-100"
                            aria-label="Válasz szerkesztése">
                        Szerk.
                    </button>
                    <!-- Ide lehetne törlés gombot is tenni később -->
                </span>
            `;
            answersList.appendChild(div);
        });
    };

// === Form Kezelő Függvények ===

    // --- Kérdés Form ---
    const resetQuestionForm = () => {
        editingQuestionId = null;
        editingQuestionIdInput.value = '';
        addQuestionForm.reset(); // Fontos a file input ürítéséhez is!
        questionGroupIdInput.value = currentGroupId || ''; // Csoport ID visszaállítása
        questionFormTitle.textContent = 'Új Kérdés Hozzáadása';
        questionSubmitButton.textContent = 'Kérdés Hozzáadása';
        questionSubmitButton.classList.remove('bg-orange-600', 'hover:bg-orange-700', 'focus:ring-orange-500');
        questionSubmitButton.classList.add('bg-green-600', 'hover:bg-green-700', 'focus:ring-green-500');
        cancelQuestionEditButton.classList.add('hidden');
        imagePreview.classList.add('hidden'); // Előnézet elrejtése
        imagePreview.src = '#'; // Előnézet src törlése
        removeImageButton.classList.add('hidden'); // Eltávolítás gomb elrejtése
        removeImageFlagInput.value = '0'; // Flag visszaállítása
        currentImageUrlInput.value = ''; // Tárolt URL törlése
        imagePreviewContainer.classList.remove('mb-2'); // Extra margó eltüntetése
    };

    const prepareQuestionFormForEdit = (questionData) => {
        resetQuestionForm(); // Kezdjük egy tiszta formmal
        editingQuestionId = questionData.id;
        editingQuestionIdInput.value = questionData.id;

        questionTextInput.value = questionData.question_text;
        questionTypeSelect.value = questionData.question_type; // Select érték beállítása

        // Kép kezelése
        if (questionData.question_image_url) {
            imagePreview.src = `${questionData.question_image_url}?t=${new Date().getTime()}`; // Cache törése
            imagePreview.classList.remove('hidden');
            removeImageButton.classList.remove('hidden');
            imagePreviewContainer.classList.add('mb-2'); // Kis hely a gombnak
            currentImageUrlInput.value = questionData.question_image_url; // URL tárolása
        } else {
            imagePreview.classList.add('hidden');
            removeImageButton.classList.add('hidden');
            currentImageUrlInput.value = '';
             imagePreviewContainer.classList.remove('mb-2');
        }
        removeImageFlagInput.value = '0'; // Alapból nem törlünk

        // Form megjelenésének módosítása
        questionFormTitle.textContent = 'Kérdés Szerkesztése';
        questionSubmitButton.textContent = 'Módosítás Mentése';
        questionSubmitButton.classList.remove('bg-green-600', 'hover:bg-green-700', 'focus:ring-green-500');
        questionSubmitButton.classList.add('bg-orange-600', 'hover:bg-orange-700', 'focus:ring-orange-500');
        cancelQuestionEditButton.classList.remove('hidden');

        questionFormSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        questionTextInput.focus();
    };    
    
    const deleteQuestion = async (questionData) => {
        if (confirm('Biztosan törlöd ezt a kérdést?\n'+questionData.question_text)) {    
            const formData = new FormData();       
            formData.set('action', 'delete_question');
            formData.set('question_id', questionData.id);
            const result = await fetchData(`${API_URL}?action=delete_question`, { method: 'POST', body:formData }); // Kérdés törlése
            if (result.success) {
               // remove html element via data-question-id
                const questionElement = document.querySelector(`[data-question-id="${questionData.id}"]`);
                questionElement.remove();
            }
        }
    };

    // --- Válasz Form Kezelése ---

    const resetAnswerForm = () => {
        editingAnswerId = null; // Szerkesztési mód kikapcsolása
        editingAnswerIdInput.value = ''; // Rejtett mező ürítése
        addAnswerForm.reset(); // Form ürítése
        answerQuestionIdInput.value = currentQuestionId || ''; // Kérdés ID visszaállítása
        answerFormTitle.textContent = 'Új Válasz Hozzáadása';
        answerSubmitButton.textContent = 'Válasz Hozzáadása';
        answerSubmitButton.classList.remove('bg-orange-600', 'hover:bg-orange-700', 'focus:ring-orange-500');
        answerSubmitButton.classList.add('bg-purple-600', 'hover:bg-purple-700', 'focus:ring-purple-500');
        cancelEditButton.classList.add('hidden'); // Mégse gomb elrejtése
    };

    const prepareFormForEdit = (answerData) => {
         editingAnswerId = answerData.id;
         editingAnswerIdInput.value = answerData.id; // Rejtett mező beállítása

         // Form mezők feltöltése
         document.getElementById('answer_text').value = answerData.text;
         document.getElementById('is_correct').checked = answerData.isCorrect === 'true' || answerData.isCorrect === true; // Kezeli a string és boolean értéket is

         // Form megjelenésének módosítása
         answerFormTitle.textContent = 'Válasz Szerkesztése';
         answerSubmitButton.textContent = 'Módosítás Mentése';
         answerSubmitButton.classList.remove('bg-purple-600', 'hover:bg-purple-700', 'focus:ring-purple-500');
         answerSubmitButton.classList.add('bg-orange-600', 'hover:bg-orange-700', 'focus:ring-orange-500'); // Más szín a szerkesztéshez
         cancelEditButton.classList.remove('hidden'); // Mégse gomb megjelenítése

         // Opcionális: Görgess a formhoz és fókuszálj az első mezőre
         answerFormSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
         document.getElementById('answer_text').focus();
    };


    // --- Adatbetöltők ---

    // loadGroups, loadQuestions (változatlanok)
    const loadGroups = async () => {
        resetQuestionForm(); // Form reset
        resetAnswerForm(); // Biztonság kedvéért ürítsük a válasz formot is
        const groups = await fetchData(`${API_URL}?action=get_groups`);
        if (groups) {
            renderGroups(groups);
        } else {
             groupsList.innerHTML = '<p class="text-red-500">Hiba a csoportok betöltésekor.</p>';
        }
        showView('groups-view');
    };

     const loadQuestions = async (groupId, groupName) => {
        resetQuestionForm(); // Form reset
        resetAnswerForm(); // Form reset
        currentGroupId = groupId;
        currentGroupName = groupName;
        selectedGroupNameEl.textContent = `- "${groupName}"`;
        questionGroupIdInput.value = groupId; // Új kérdéshez a csoport ID

        const questions = await fetchData(`${API_URL}?action=get_questions&group_id=${groupId}`);
         if (questions) {
            renderQuestions(questions);
        } else {
             questionsList.innerHTML = '<p class="text-red-500">Hiba a kérdések betöltésekor.</p>';
        }
        showView('questions-view');
    };


    const loadAnswers = async (questionId, questionText) => {
        // Nem kell a question formot resetelni, ha csak válaszokat nézünk
        resetAnswerForm(); // Csak a válasz formot reseteljük
        currentQuestionId = questionId;
        currentQuestionText = questionText;
        // selectedQuestionTextEl.textContent = `"${questionText}"`;
	    selectedQuestionTextEl.innerHTML = `"${questionText}"`;
        answerQuestionIdInput.value = questionId;

        const answers = await fetchData(`${API_URL}?action=get_answers&question_id=${questionId}`);
         if (answers) {
            renderAnswers(answers);
        } else {
             answersList.innerHTML = '<p class="text-red-500">Hiba a válaszok betöltésekor.</p>';
        }
        showView('answers-view');
    };

    // --- Eseménykezelők ---

    // addGroupForm submit (változatlan)
    addGroupForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData(addGroupForm);
        const data = {
            group_name: formData.get('group_name'),
            release_date: formData.get('release_date')
        };

        const result = await fetchData(`${API_URL}?action=add_group`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });

        if (result?.id) { // Biztonságosabb ellenőrzés
            showMessage('Kérdéscsoport sikeresen hozzáadva.');
            addGroupForm.reset();
            loadGroups();
        }
    });

     // MÓDOSÍTOTT: addQuestionForm submit (FormData küldése)
    addQuestionForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        // FormData objektum létrehozása a formból
        const formData = new FormData(addQuestionForm);

        // Biztosítjuk, hogy a group_id is benne legyen (ha esetleg hiányozna a formból)
        if (!formData.has('group_id') && currentGroupId) {
             formData.set('group_id', currentGroupId);
        }
         // Ha üres a file input, töröljük a FormData-ból, hogy a backend ne kapja meg üresen
         const imageFile = formData.get('question_image_file');
         if (imageFile && imageFile.size === 0) {
             formData.delete('question_image_file');
         }


        let apiUrl;
        let successMessage;

        if (editingQuestionId) {
            // --- SZERKESZTÉS ---
            apiUrl = `${API_URL}?action=update_question`; // Vagy action=update_question a FormData-ban
            // Az editing_question_id már benne van a FormData-ban a hidden input miatt ('question_id' néven)
             formData.set('action', 'update_question'); // Action hozzáadása a FormData-hoz is lehet
            successMessage = 'Kérdés sikeresen módosítva.';
        } else {
            // --- HOZZÁADÁS ---
             if (!formData.get('group_id')) {
                 showMessage('Hiba: Hiányzó csoport azonosító!', true);
                 return;
             }
            apiUrl = `${API_URL}?action=add_question`;
             formData.set('action', 'add_question');
            successMessage = 'Kérdés sikeresen hozzáadva.';
        }

        // Fontos: a fetchData automatikusan kezeli a FormData-t és a headereket
        const result = await fetchData(apiUrl, {
            method: 'POST',
            body: formData // Itt a FormData objektumot küldjük
            // HEADERS NEM KELLENEK!
        });

        if (result?.id) { // Várjuk, hogy legyen ID a válaszban
            showMessage(result.message || successMessage);
            resetQuestionForm(); // Form ürítése és visszaállítása
            loadQuestions(currentGroupId, currentGroupName); // Lista frissítése
        }
        // Hiba esetén a fetchData már kezeli az üzenetet
    });

    // Kérdés Form: Fájl kiválasztásának figyelése (előnézet)
    questionImageFileInput.addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (file && file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = (event) => {
                imagePreview.src = event.target.result;
                imagePreview.classList.remove('hidden');
                removeImageButton.classList.remove('hidden'); // Megjelenítjük a törlés gombot is
                 removeImageFlagInput.value = '0'; // Ha új képet választ, ne törölje
                 imagePreviewContainer.classList.add('mb-2');
            }
            reader.readAsDataURL(file);
        } else {
            // Ha nem képfájlt választott, vagy törölte a kiválasztást
            imagePreview.classList.add('hidden');
            imagePreview.src = '#';
            // A remove gombot csak akkor rejtjük el, ha nem szerkesztési módban volt kép
            if (!currentImageUrlInput.value) {
                 removeImageButton.classList.add('hidden');
                 imagePreviewContainer.classList.remove('mb-2');
            }
        }
    });

    // Kérdés Form: Kép eltávolítása gomb
    removeImageButton.addEventListener('click', () => {
        imagePreview.classList.add('hidden');
        imagePreview.src = '#';
        removeImageButton.classList.add('hidden');
        questionImageFileInput.value = ''; // Fontos: ürítjük a file inputot!
        removeImageFlagInput.value = '1'; // Beállítjuk a törlési flag-et
        currentImageUrlInput.value = ''; // Töröljük a tárolt URL-t is
        imagePreviewContainer.classList.remove('mb-2');
    });

    // Kérdés Form: Mégse gomb
    cancelQuestionEditButton.addEventListener('click', () => {
        resetQuestionForm();
    });

    // ÚJ: Eseménykezelő a kérdéslistához (Szerkesztés / Válaszok gombok)
    questionsList.addEventListener('click', (e) => {
        const target = e.target;
        const questionItem = target.closest('.question-item');
        if (!questionItem) return; // Nem kérdés elemen belüli kattintás

        const questionData = JSON.parse(questionItem.dataset.questionData || '{}');
        if (!questionData.id) return; // Nincs ID

        // Szerkesztés gombra kattintott?
        if (target.classList.contains('edit-question-btn')) {
            prepareQuestionFormForEdit(questionData);
        }
        // Válaszok gombra kattintott?
        else if (target.classList.contains('view-answers-btn')) {
            loadAnswers(questionData.id, questionData.question_text);
        }
        // Törlés gombra kattintott?
        else if (target.classList.contains('delete-question-btn')) {
            deleteQuestion(questionData);
        }
    });


    // MÓDOSÍTOTT: addAnswerForm submit (hozzáadás VAGY szerkesztés)
    addAnswerForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData(addAnswerForm);
        const isCorrectValue = formData.get('is_correct') === 'on'; // Checkbox -> boolean

        let apiUrl;
        let data;
        let successMessage;

        if (editingAnswerId) {
            // --- SZERKESZTÉS ---
            apiUrl = `${API_URL}?action=update_answer`;
            data = {
                id: editingAnswerId, // A szerkesztett válasz ID-ja
                answer_text: formData.get('answer_text'),
                is_correct: isCorrectValue
                // question_id nem kell a szervernek update-hez, de validáláshoz jó lehetne
            };
            successMessage = 'Válasz sikeresen módosítva.';

        } else {
            // --- HOZZÁADÁS ---
            apiUrl = `${API_URL}?action=add_answer`;
            data = {
                question_id: parseInt(formData.get('question_id')),
                answer_text: formData.get('answer_text'),
                is_correct: isCorrectValue
            };
             if (!data.question_id || isNaN(data.question_id)) {
                 showMessage('Hiba: Érvénytelen vagy hiányzó kérdés azonosító!', true);
                 return;
            }
            successMessage = 'Válasz sikeresen hozzáadva.';
        }

         const result = await fetchData(apiUrl, {
            method: 'POST', // Maradhat POST mindkét esetben az API egyszerűsége miatt
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });

        if (result?.id || result?.success) { // Ellenőrizzük az ID-t vagy egy 'success' flag-et
            showMessage(result.message || successMessage); // Használjuk a szerver üzenetét, ha van
            resetAnswerForm(); // Form visszaállítása alaphelyzetbe
            loadAnswers(currentQuestionId, currentQuestionText); // Lista frissítése
        }
         // Hiba esetén a fetchData már kezeli az üzenetet
    });

    // ÚJ: Eseménykezelő a válaszlistában a "Szerkesztés" gombokhoz (event delegation)
    answersList.addEventListener('click', (e) => {
        // Ellenőrizzük, hogy a '.edit-answer-btn' osztályú gombon történt-e a kattintás
        if (e.target && e.target.classList.contains('edit-answer-btn')) {
            const answerItem = e.target.closest('.answer-item'); // Megkeressük a szülő válasz div-et
            if (answerItem && answerItem.dataset.answerId) {
                 const answerData = {
                     id: parseInt(answerItem.dataset.answerId), // Legyen szám
                     text: answerItem.dataset.answerText,
                     isCorrect: answerItem.dataset.isCorrect // Ez string lesz ("true"/"false")
                 };
                 prepareFormForEdit(answerData);
            } else {
                console.error("Nem található a válasz elem vagy annak data attribútumai.");
                showMessage("Hiba történt a szerkesztés előkészítésekor.", true);
            }
        }
    });

    // ÚJ: Eseménykezelő a "Mégse" gombhoz
    cancelEditButton.addEventListener('click', () => {
        resetAnswerForm();
    });


    // Vissza gombok (változatlanok)
    backToGroupsBtn.addEventListener('click', () => {
        // A nézetváltó már kezeli a form resetet
        // currentGroupId = null;
        // currentGroupName = '';
        showView('groups-view');
    });

    backToQuestionsBtn.addEventListener('click', () => {
         // A nézetváltó már kezeli a form resetet
        // currentQuestionId = null;
        // currentQuestionText = '';
        if (currentGroupId) {
           showView('questions-view');
        } else {
             loadGroups();
        }
    });


    // --- Inicializálás ---
    loadGroups();

});