#!/usr/bin/env php
<?php
/**
 * Push Notification Worker Script
 *
 * Processes queued push notification jobs (original and reminders)
 * and handles scheduled automatic retries for failed deliveries.
 *
 * Run via Cron or Supervisor.
 */

// --- Worker Configuration & Setup ---
// Ensure this script can run via PHP CLI


// Set timezone if needed
date_default_timezone_set('Europe/Budapest');

// Increase execution time and memory for CLI script if necessary
ini_set('max_execution_time', 0); // 0 = Unlimited for CLI
ini_set('memory_limit', '512M'); // Adjust as needed

// --- Include Shared Config/Functions ---
require_once __DIR__ . '/config.php';

// --- Include Necessary Functions ---
function db_connect()
{
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        error_log("WORKER: Database Connection Error: " . $conn->connect_error);
        exit(1); // Exit worker if DB connection fails
    }
    $conn->set_charset("utf8mb4");
    return $conn;
}



/**
 * Sends a single push notification via FCM v1 API.
 *
 * @param string $token FCM device registration token.
 * @param string $title Notification title.
 * @param string $description Notification body.
 * @return array Associative array with 'success' (bool), 'error' (string|null), 'response' (array|null), 'http_code' (int|null).
 */
function send_fcm_notification(string $token, string $title, string $description): array
{
    $payload = [
        'message' => [
            'token' => $token,
            'notification' => [
                'title' => $title,
                'body' => $description,
            ],
        ]
    ];
    $headers = [
        'Authorization: Bearer ' . FCM_SERVER_KEY,
        'Content-Type: application/json'
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, FCM_API_URL);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Consider true in production
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

    $result = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    $response_data = $result ? json_decode($result, true) : null;

    if ($curl_error) {
        return ['success' => false, 'error' => 'Curl error: ' . $curl_error, 'response' => $response_data, 'http_code' => $http_code];
    }

    if ($http_code === 200 && isset($response_data['name'])) {
        return ['success' => true, 'error' => null, 'response' => $response_data, 'http_code' => $http_code];
    } else {
        $error_message = 'FCM API error';
        if (isset($response_data['error']['message'])) {
            $error_message = $response_data['error']['message'];
            if (isset($response_data['error']['status'])) {
                $error_message .= ' (Status: ' . $response_data['error']['status'] . ')';
            }
        } elseif ($result) {
            $error_message = 'FCM API error (non-JSON or unexpected response): ' . substr($result, 0, 200);
        }
        if ($http_code && $http_code !== 200) {
            $error_message .= " (HTTP Code: {$http_code})";
        }
        return ['success' => false, 'error' => $error_message, 'response' => $response_data, 'http_code' => $http_code];
    }
}

/**
 * Marks a user's token as invalid in the database (sets token to NULL).
 *
 * @param mysqli $db Database connection.
 * @param int $userId The ID of the user.
 * @param string $token The token to nullify.
 * @return void
 */
function unsubscribe_fcm_token(mysqli $db, int $userId, string $token): void
{
    $stmt = $db->prepare("UPDATE inspiracio_fbuser SET token = NULL WHERE id = ? AND token = ?");
    if ($stmt) {
        $stmt->bind_param('is', $userId, $token);
        if (!$stmt->execute()) {
            error_log("WORKER: Failed to nullify token for user ID {$userId}: " . $stmt->error);
        } else {
            if ($stmt->affected_rows > 0) {
                echo "WORKER: Unsubscribed token for user ID {$userId}.\n";
            }
        }
        $stmt->close();
    } else {
        error_log("WORKER: Failed to prepare statement to nullify token for user ID {$userId}: " . $db->error);
    }
}

/**
 * Fetches target users based on 'all' or 'specific' emails criteria. (PHP 7.3 Compatible)
 *
 * @param mysqli $db Database connection.
 * @param string $targetType Either 'all', 'emw_nincs_jegye' or 'specific'.
 * @param string|null $targetEmailsInput Comma-separated emails if targetType is 'specific'.
 * @return array Array of users, each user is ['id' => int, 'mail' => string, 'token' => string]. Returns empty array on error or no users found.
 */
function get_target_users(mysqli $db, string $targetType, ?string $targetEmailsInput): array
{
    $targetUsers = [];
    try {
        if ($targetType === 'all') {
            $sql = "SELECT id, mail, token FROM inspiracio_fbuser WHERE token IS NOT NULL AND token != '' AND mester_app_login > 0 AND unregistered = 0";
            $result = $db->query($sql);
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $targetUsers[] = $row;
                }
                $result->free();
            } else {
                throw new Exception("Failed to query all users: " . $db->error);
            }
        } elseif ($targetType === 'emw_nincs_jegye') {
            $sql = "SELECT user.id, user.mail, user.token 
            FROM dasl1_db_inspiracio.inspiracio_fbuser AS user 
            WHERE 
                user.token IS NOT NULL 
                AND user.token != '' 
                AND user.mester_app_login > 0 
                AND user.unregistered = 0 
                AND (user.mail IS NOT NULL AND user.mail != '') 
                AND NOT EXISTS ( SELECT 1 FROM dasl1_db_jegyvasarlas.jegyek j WHERE j.user_hashAuth = user.hashAuth )
            ";
            $result = $db->query($sql);
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $targetUsers[] = $row;
                }
                $result->free();
            } else {
                throw new Exception("Failed to query emw_nincs_jegye users: " . $db->error);
            }
        } elseif ($targetType === 'specific') {
            if (empty($targetEmailsInput)) {
                echo "WORKER: No target emails provided for specific type.\n";
                return [];
            }
            $targetEmailsArray = array_map('trim', explode(',', $targetEmailsInput));
            // --- CORRECTED LINE FOR PHP 7.3 ---
            $targetEmailsArray = array_filter($targetEmailsArray, function ($email) {
                return filter_var($email, FILTER_VALIDATE_EMAIL);
            });
            // --- END CORRECTION ---

            if (empty($targetEmailsArray)) {
                echo "WORKER: No valid target emails found after filtering.\n";
                return [];
            }

            $placeholders = implode(',', array_fill(0, count($targetEmailsArray), '?'));
            $types = str_repeat('s', count($targetEmailsArray));
            $sql = "SELECT id, mail, token FROM inspiracio_fbuser WHERE mail IN ($placeholders) AND token IS NOT NULL AND token != ''";

            $stmt = $db->prepare($sql);
            if (!$stmt) {
                throw new Exception("Prepare statement failed (get specific users): " . $db->error);
            }
            $stmt->bind_param($types, ...$targetEmailsArray);
            if (!$stmt->execute()) {
                throw new Exception("Execute failed (get specific users): " . $stmt->error);
            }
            // $result = $stmt->get_result();
            //while ($row = $result->fetch_assoc()) {
            //    $targetUsers[] = $row;
            //}
			$stmt->bind_result($id, $mail, $token);
			$targetUsers = [];

			while ($stmt->fetch()) {
				$targetUsers[] = [
					'id'    => $id,
					'mail'  => $mail,
					'token' => $token
				];
			}
            $stmt->close();
        }
    } catch (Exception $e) {
        error_log("WORKER: Error fetching target users (Type: {$targetType}): " . $e->getMessage());
        return [];
    }
    return $targetUsers;
}


/**
 * Fetches users for the specific quiz reminder logic.
 * Executes the predefined SQL query.
 *
 * @param mysqli $db Database connection.
 * @return array Array of users matching the criteria, each ['id', 'mail', 'token']. Returns empty array on error or no users found.
 */
function get_quiz_reminder_users(mysqli $db): array
{
    echo "WORKER: Fetching users for quiz reminder...\n";
    $targetUsers = [];
    try {
        // Set the session variable for the latest group ID within the transaction/connection
        $result_var = $db->query("SET @latest_group_id = (SELECT id FROM quiz_question_groups ORDER BY release_date DESC, id DESC LIMIT 1)");
        if (!$result_var) {
            throw new Exception("Failed to set @latest_group_id: " . $db->error);
        }

        // The main selection query
        $sql = "
            SELECT DISTINCT
                u.id,
                u.mail,
                u.token
            FROM
                inspiracio_fbuser u
            JOIN
                quiz_user_answers qa_any ON u.id = qa_any.fbuser_id
            LEFT JOIN
                (
                    SELECT DISTINCT qa_latest.fbuser_id
                    FROM quiz_user_answers qa_latest
                    JOIN quiz_questions qq_latest ON qa_latest.question_id = qq_latest.id
                    WHERE qq_latest.question_group_id = @latest_group_id
                ) AS answered_latest ON u.id = answered_latest.fbuser_id
            WHERE
                answered_latest.fbuser_id IS NULL
            AND
                EXISTS (
                    SELECT 1
                    FROM
                        quiz_user_answers qa_last4
                    JOIN
                        quiz_questions qq_last4 ON qa_last4.question_id = qq_last4.id
                    JOIN
                        (
                            SELECT id
                            FROM quiz_question_groups
                            ORDER BY release_date DESC, id DESC
                            LIMIT 4
                        ) AS last_four_groups ON qq_last4.question_group_id = last_four_groups.id
                    WHERE
                        qa_last4.fbuser_id = u.id
                )
             AND u.token IS NOT NULL AND u.token != '' -- Crucial: Ensure token exists
        ";

        $result = $db->query($sql);
        if (!$result) {
            throw new Exception("Failed to execute main quiz user query: " . $db->error);
        }

        while ($row = $result->fetch_assoc()) {
            $targetUsers[] = $row;
        }
        $result->free();
    } catch (Exception $e) {
        error_log("WORKER: Failed to get quiz reminder users: " . $e->getMessage());
        return []; // Return empty on error
    }
    echo "WORKER: Found " . count($targetUsers) . " users matching quiz reminder criteria.\n";
    return $targetUsers;
}

/**
 * Schedules an automatic retry for a failed delivery.
 * Inserts a 'pending' record into the push_notification_retries table.
 *
 * @param mysqli $db Database connection.
 * @param int $original_delivery_status_id The ID of the failed record in push_notification_delivery_status.
 * @param int $logId The ID of the original batch job log.
 * @param int|null $userId The user ID associated with the failed token.
 * @param string|null $email The email associated with the failed token.
 * @param string $token The token that failed.
 * @return void
 */
function schedule_retry(mysqli $db, int $original_delivery_status_id, int $logId, ?int $userId, ?string $email, string $token): void
{
    if (empty($token) || strlen($token) < 20) {
        echo "WORKER: Skipping retry schedule for invalid/empty token (User ID: {$userId}, Delivery ID: {$original_delivery_status_id}).\n";
        return;
    }

    echo "WORKER: Scheduling retry for delivery ID {$original_delivery_status_id} from log {$logId}\n";
    try {
        $retryTime = (new DateTime('now', new DateTimeZone('UTC')))
            ->add(new DateInterval('P1D'))
            ->format('Y-m-d H:i:s');

        $stmt = $db->prepare(
            "INSERT INTO push_notification_retries
                (original_delivery_status_id, push_notification_log_id, target_user_id, target_email, target_token, scheduled_retry_at, retry_status)
             VALUES (?, ?, ?, ?, ?, ?, 'pending')"
        );
        if (!$stmt) {
            throw new Exception("Prepare failed (schedule retry): " . $db->error);
        }
        $stmt->bind_param('iiisss', $original_delivery_status_id, $logId, $userId, $email, $token, $retryTime);
        if (!$stmt->execute()) {
            if ($db->errno === 1062) {
                echo "WORKER: Retry for delivery ID {$original_delivery_status_id} already scheduled.\n";
            } else {
                throw new Exception("Execute failed (schedule retry): " . $stmt->error);
            }
        } else {
            echo "WORKER: Retry scheduled for {$retryTime} (Retry Record ID: {$stmt->insert_id})\n";
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("WORKER: Failed to schedule retry for delivery ID {$original_delivery_status_id}: " . $e->getMessage());
    }
}

/**
 * Processes pending retry jobs that are due.
 * Finds, locks, processes, and updates status for retry attempts.
 *
 * @param mysqli $db Database connection.
 * @return int Number of retries processed in this call.
 */
function process_pending_retries(mysqli $db): int
{
    // echo "WORKER: Checking for pending retries...\n"; // Can be noisy
    $retries_processed_this_run = 0;
    $max_retries_per_run = 100;

    while ($retries_processed_this_run < $max_retries_per_run) {
        $retry_id = null;
        $retry_data = null;
        $lockAcquired = false;

        $db->begin_transaction();
        try {
            $sqlFindRetry = "SELECT r.id, r.push_notification_log_id, r.target_user_id, r.target_email, r.target_token,
                                    l.title, l.description
                             FROM push_notification_retries r
                             JOIN push_notification_logs l ON r.push_notification_log_id = l.id
                             WHERE r.retry_status = 'pending'
                               AND r.scheduled_retry_at <= NOW()
                             ORDER BY r.scheduled_retry_at ASC
                             LIMIT 1
                             FOR UPDATE";

            $stmtFindRetry = $db->prepare($sqlFindRetry);
            if (!$stmtFindRetry) throw new Exception("Prepare failed (find retry): " . $db->error);
            if (!$stmtFindRetry->execute()) throw new Exception("Execute failed (find retry): " . $stmtFindRetry->error);
            //$resultRetry = $stmtFindRetry->get_result();
            //$retry_data = $resultRetry->fetch_assoc();
            // --- MODIFICATION START ---
            // Bind result variables
            $stmtFindRetry->bind_result(
                $r_id,
                $r_push_notification_log_id,
                $r_target_user_id,
                $r_target_email,
                $r_target_token,
                $l_title,
                $l_description
            );

            $retry_data = null;
            if ($stmtFindRetry->fetch()) { // Fetch the first (and only) row
                $retry_data = [
                    'id' => $r_id,
                    'push_notification_log_id' => $r_push_notification_log_id,
                    'target_user_id' => $r_target_user_id,
                    'target_email' => $r_target_email,
                    'target_token' => $r_target_token,
                    'title' => $l_title,
                    'description' => $l_description
                ];
            }
            // --- MODIFICATION END ---	
            $stmtFindRetry->close();

            if ($retry_data) {
                $retry_id = $retry_data['id'];

                $stmtUpdateRetry = $db->prepare("UPDATE push_notification_retries SET retry_status = 'processing' WHERE id = ? AND retry_status = 'pending'");
                if (!$stmtUpdateRetry) throw new Exception("Prepare failed (mark retry processing): " . $db->error);
                $stmtUpdateRetry->bind_param('i', $retry_id);
                if (!$stmtUpdateRetry->execute()) throw new Exception("Execute failed (mark retry processing): " . $stmtUpdateRetry->error);
                if ($stmtUpdateRetry->affected_rows == 0) throw new Exception("Retry {$retry_id} status wasn't pending or already locked.");
                $stmtUpdateRetry->close();

                $db->commit();
                $lockAcquired = true;
                // echo "WORKER: Locked and marked retry {$retry_id} as processing.\n"; // Can be noisy
            } else {
                $db->commit();
                break;
            }
        } catch (Exception $e) {
            $db->rollback();
            error_log("WORKER: Error finding/locking retry job: " . $e->getMessage());
            break;
        }

        if ($lockAcquired && $retry_id && $retry_data) {
            $retry_status = 'failure';
            $retry_error_message = null;
            $retry_should_unsubscribe = false;

            try {
                echo "WORKER: Processing retry {$retry_id} for token starting " . substr($retry_data['target_token'], 0, 10) . "...\n";

                $fcmResult = send_fcm_notification(
                    $retry_data['target_token'],
                    $retry_data['title'],
                    $retry_data['description']
                );

                if ($fcmResult['success']) {
                    $retry_status = 'success';
                    echo "WORKER: Retry {$retry_id} successful.\n";
                } else {
                    $retry_status = 'failure';
                    $retry_error_message = $fcmResult['error'] ?? 'Unknown FCM error during retry';
                    echo "WORKER: Retry {$retry_id} failed: {$retry_error_message}\n";

                    $fcmErrorString = $retry_error_message ?: '';
                    $fcmResponseData = $fcmResult['response'] ?? [];
                    $fcmErrorStatus = $fcmResponseData['error']['status'] ?? null;
                    $fcmErrorMessageLower = strtolower($fcmErrorString);

                    if ($fcmErrorStatus === 'UNREGISTERED' || /* Add other conditions */ false) {
                        $retry_should_unsubscribe = true;
                        $retry_error_message .= ' (Token Marked Invalid on Retry)';
                    }
                }

                if ($retry_should_unsubscribe && isset($retry_data['target_user_id']) && $retry_data['target_user_id'] > 0) {
                    try {
                        unsubscribe_fcm_token($db, $retry_data['target_user_id'], $retry_data['target_token']);
                    } catch (Exception $e) {
                        error_log("WORKER: Failed unsubscribe during retry {$retry_id}: {$e->getMessage()}");
                    }
                } else if ($retry_should_unsubscribe) {
                    error_log("WORKER: Cannot unsubscribe token after retry {$retry_id} because target_user_id is missing.");
                }
            } catch (Exception $e) {
                error_log("WORKER: CRITICAL Error processing retry {$retry_id}: " . $e->getMessage());
                $retry_status = 'failure';
                $retry_error_message = "Worker error during retry processing: " . $e->getMessage();
            }

            try {
                $db->begin_transaction();
                $stmtUpdateFinalRetry = $db->prepare("UPDATE push_notification_retries SET retry_status = ?, retry_fcm_error_message = ?, retry_processed_at = NOW() WHERE id = ?");
                if (!$stmtUpdateFinalRetry) throw new Exception("Prepare failed (update final retry status): " . $db->error);
                $stmtUpdateFinalRetry->bind_param('ssi', $retry_status, $retry_error_message, $retry_id);
                if (!$stmtUpdateFinalRetry->execute()) throw new Exception("Execute failed (update final retry status): " . $stmtUpdateFinalRetry->error);
                $stmtUpdateFinalRetry->close();
                $db->commit();
                $retries_processed_this_run++;
            } catch (Exception $e) {
                $db->rollback();
                error_log("WORKER: FAILED to update final status for retry {$retry_id}: " . $e->getMessage());
                break;
            }
        }
    }
    // echo "WORKER: Finished checking/processing retries for this run ({$retries_processed_this_run} processed).\n"; // Can be noisy
    return $retries_processed_this_run;
}


// --- ================= Worker Main Execution ================= ---

echo "WORKER: Script started at: " . date('Y-m-d H:i:s') . "\n";
$db = db_connect();
$original_job_processed = false;

// Valahol van egy bug ami miatt több pontot érnek el. Többszőr adják fel a válaszokat. Ez törli az azonosakat
$sql = "DELETE q1 FROM quiz_user_answers q1 JOIN quiz_user_answers q2 ON q1.fbuser_id = q2.fbuser_id AND q1.question_id = q2.question_id AND q1.answer_id = q2.answer_id AND q1.id > q2.id;";
$result = $db->query($sql);

// --- === Phase 1: Process ONE Original/Reminder Job === ---
$logId = null;
$jobData = null;
$lockAcquired = false;

echo "WORKER: Checking for original/reminder jobs...\n";
$db->begin_transaction();
try {
    $sqlFindJob = "SELECT id, title, description, target_type, target_emails
                   FROM push_notification_logs
                   WHERE fcm_response_status = 'queued'
                     AND (scheduled_send_at IS NULL OR scheduled_send_at <= NOW())
                   ORDER BY scheduled_send_at ASC, sent_at ASC
                   LIMIT 1
                   FOR UPDATE";

    $stmtFindJob = $db->prepare($sqlFindJob);
    if (!$stmtFindJob) throw new Exception("Prepare failed (find job): " . $db->error);
    if (!$stmtFindJob->execute()) throw new Exception("Execute failed (find job): " . $stmtFindJob->error);
    //$resultJob = $stmtFindJob->get_result();
    //$jobData = $resultJob->fetch_assoc();
    // --- MODIFICATION START ---
    // Bind result variables
    $stmtFindJob->bind_result($id, $title, $description, $target_type, $target_emails);

    $jobData = null;
    if ($stmtFindJob->fetch()) { // Fetch the first (and only) row
        $jobData = [
            'id' => $id,
            'title' => $title,
            'description' => $description,
            'target_type' => $target_type,
            'target_emails' => $target_emails,
        ];
    }
    // --- MODIFICATION END ---	
    $stmtFindJob->close();

    if ($jobData) {
        $logId = $jobData['id'];
        echo "WORKER: Found eligible job ID: {$logId} (Type: {$jobData['target_type']})\n";

        $stmtUpdateStatus = $db->prepare("UPDATE push_notification_logs SET fcm_response_status = 'processing' WHERE id = ? AND fcm_response_status = 'queued'");
        if (!$stmtUpdateStatus) throw new Exception("Prepare failed (mark job processing): " . $db->error);
        $stmtUpdateStatus->bind_param('i', $logId);
        if (!$stmtUpdateStatus->execute()) throw new Exception("Execute failed (mark job processing): " . $stmtUpdateStatus->error);
        if ($stmtUpdateStatus->affected_rows == 0) throw new Exception("Job {$logId} status wasn't 'queued' or already locked.");
        $stmtUpdateStatus->close();

        $db->commit();
        $lockAcquired = true;
        echo "WORKER: Locked and marked job {$logId} as processing.\n";
    } else {
        $db->commit();
        echo "WORKER: No original/reminder jobs found eligible at this time.\n";
    }
} catch (Exception $e) {
    $db->rollback();
    error_log("WORKER: Error finding/locking original/reminder job: " . $e->getMessage());
}

// --- Process the Acquired Original/Reminder Job ---
if ($lockAcquired && $logId && $jobData) {
    $currentSuccessCount = 0;
    $currentFailureCount = 0;
    $currentUnsubscribedCount = 0;
    $finalStatus = 'failure';
    $processingError = null;
    $targetUsers = [];

    try {
        $title = $jobData['title'];
        $description = $jobData['description'];
        $targetType = $jobData['target_type'];
        $targetEmailsInput = $jobData['target_emails'];

        echo "WORKER: Fetching target users for job {$logId} (Type: {$targetType})...\n";
        if ($targetType === 'all' || $targetType === 'specific' || $targetType === 'emw_nincs_jegye') {
            $targetUsers = get_target_users($db, $targetType, $targetEmailsInput);
        } elseif ($targetType === 'quiz_inactive_recent') {
            $targetUsers = get_quiz_reminder_users($db);
        } else {
            throw new Exception("Unknown target_type encountered: '{$targetType}'");
        }
        $totalTargets = count($targetUsers);
        echo "WORKER: Found {$totalTargets} target users for job {$logId}.\n";

        if ($totalTargets === 0) {
            $finalStatus = 'no_targets';
            echo "WORKER: No valid targets found for job {$logId}. Marking as 'no_targets'.\n";
        } else {
            $stmtDetail = $db->prepare("INSERT INTO push_notification_delivery_status (push_notification_log_id, target_user_id, target_email, target_token, status, fcm_error_message) VALUES (?, ?, ?, ?, ?, ?)");
            if (!$stmtDetail) throw new Exception("Prepare failed (detail log): " . $db->error);

            foreach ($targetUsers as $user) {
                $userId = $user['id'] ?? null;
                $email = $user['mail'] ?? null;
                $token = $user['token'] ?? null;

                $attemptStatus = 'failure';
                $attemptErrorMessage = 'Processing skipped or failed before FCM call';
                $shouldUnsubscribe = false;
                $deliveryDetailId = null;

                try {
                    if (empty($token)) {
                        $attemptErrorMessage = 'User token was missing in database';
                    } else {
                        $fcmResult = send_fcm_notification($token, $title, $description);
                        $isSuccess = $fcmResult['success'];
                        $attemptErrorMessage = $fcmResult['error'] ?? null;

                        if ($isSuccess) {
                            $attemptStatus = 'success';
                        } else {
                            $attemptStatus = 'failure';
                            $fcmErrorString = $attemptErrorMessage ?: '';
                            $fcmResponseData = $fcmResult['response'] ?? [];
                            $fcmErrorStatus = $fcmResponseData['error']['status'] ?? null;
                            $fcmErrorMessageLower = strtolower($fcmErrorString);

                            if ($fcmErrorStatus === 'UNREGISTERED' || /* Add other conditions */ false) {
                                $shouldUnsubscribe = true;
                                $attemptErrorMessage .= ' (Token Marked Invalid)';
                            }
                        }
                    }
                } catch (Exception $e) {
                    $attemptStatus = 'failure';
                    $attemptErrorMessage = "Error during processing token: " . $e->getMessage();
                    error_log("WORKER: Error processing user ID {$userId} / token {$token} in job {$logId}: " . $e->getMessage());
                }

                if ($attemptStatus === 'success') {
                    $currentSuccessCount++;
                } else {
                    $currentFailureCount++;
                }

                if ($shouldUnsubscribe) {
                    if ($userId) {
                        try {
                            unsubscribe_fcm_token($db, $userId, $token);
                            $currentUnsubscribedCount++;
                        } catch (Exception $unsubError) {
                            error_log("WORKER: Error during unsubscribe for user ID {$userId} in job {$logId}: " . $unsubError->getMessage());
                        }
                    } else {
                        error_log("WORKER: Cannot unsubscribe token after failure in job {$logId} because target_user_id is missing for token starting " . substr($token ?? '', 0, 10));
                    }
                }

                try {
                    $stmtDetail->bind_param('iissss', $logId, $userId, $email, $token, $attemptStatus, $attemptErrorMessage);
                    if (!$stmtDetail->execute()) {
                        error_log("WORKER: Failed to log detail for user {$userId} / token {$token} (Job ID: {$logId}): " . $stmtDetail->error);
                    } else {
                        $deliveryDetailId = $stmtDetail->insert_id;
                    }
                } catch (Exception $e) {
                    error_log("WORKER: Exception logging detail for {$userId}: {$e->getMessage()}");
                }


                if ($attemptStatus === 'failure' && $deliveryDetailId) {
                    if (strpos($attemptErrorMessage ?? '', 'token was missing') === false) {
                        schedule_retry($db, $deliveryDetailId, $logId, $userId, $email, $token);
                    }
                }
                // usleep(50000); // Optional delay
            }
            $stmtDetail->close();

            if ($currentSuccessCount > 0 && $currentFailureCount == 0) $finalStatus = 'success';
            elseif ($currentSuccessCount > 0 && $currentFailureCount > 0) $finalStatus = 'partial_failure';
            elseif ($currentSuccessCount == 0 && $currentFailureCount > 0) $finalStatus = 'failure';
            else $finalStatus = 'failure';
        }
        echo "WORKER: Finished processing job {$logId}. Status: {$finalStatus}, Success: {$currentSuccessCount}, Fail: {$currentFailureCount}, Unsub: {$currentUnsubscribedCount}\n";
    } catch (Exception $e) {
        error_log("WORKER: CRITICAL Error processing job {$logId}: " . $e->getMessage());
        $processingError = $e->getMessage();
        $finalStatus = 'failure';
    }

    if ($logId) {
        try {
            $db->begin_transaction();
            $stmtLogUpdate = $db->prepare("UPDATE push_notification_logs SET success_count = ?, failure_count = ?, unsubscribed_count = ?, fcm_response_status = ?, fcm_response_details = ? WHERE id = ?");
            if (!$stmtLogUpdate) throw new Exception("Prepare failed (final update log): " . $db->error);
            $stmtLogUpdate->bind_param('iiissi', $currentSuccessCount, $currentFailureCount, $currentUnsubscribedCount, $finalStatus, $processingError, $logId);
            if (!$stmtLogUpdate->execute()) throw new Exception("Execute failed (final update log): " . $stmtLogUpdate->error);
            $stmtLogUpdate->close();
            $db->commit();
            echo "WORKER: Successfully updated final status for job {$logId}.\n";
            $original_job_processed = true;
        } catch (Exception $e) {
            $db->rollback();
            error_log("WORKER: FAILED to update final status for job {$logId} after processing: " . $e->getMessage());
        }
    }
}

// --- === Phase 2: Process Pending Retries === ---
try {
    $retries_done = process_pending_retries($db);
    // echo "WORKER: Processed {$retries_done} retry attempts.\n"; // Can be noisy
} catch (Exception $e) {
    error_log("WORKER: Uncaught error during retry processing phase: " . $e->getMessage());
}


// --- Worker Finish ---
if ($original_job_processed) {
    echo "WORKER: Finished processing an original/reminder job.\n";
}
echo "WORKER: Script finished run at: " . date('Y-m-d H:i:s') . "\n";
$db->close();
exit(0);

?>