/**
 * Normalize Hungarian numbers to E.164.
 * For non-HU numbers:
 *  - If input starts with '+' or '00', we keep it non-destructively for display,
 *    but we still produce a compact E.164 (digits only after '+') for storage.
 *  - We DO NOT try to guess country if no '+'/'00' and it's not HU-like.
 * Returns { e164, displayHint } or { e164: null, displayHint: null } if invalid.
 *
 * displayHint:
 *  - HU numbers → pretty formatted "+36 ..." string.
 *  - non-HU numbers → the user's ORIGINAL input (trimmed), preserving spaces/punctuation.
 */
function normalizePhone(input) {
  
  if (input == null) return { e164: null, displayHint: null };

  const original = String(input).trim();
  if (!original) return { e164: null, displayHint: null };

  // Helper: compact to E.164 form +digits
  const toCompactE164 = (s) => {
    // keep leading +, strip all non-digits
    const compact = "+" + s.replace(/[^\d]/g, "");
    // ITU max length 15 digits (excluding '+'); min ~6 to be sane
    if (!/^\+\d{6,15}$/.test(compact)) return null;
    return compact;
  };

  // 1) If looks explicitly international (starts with + or 00)
  if (original.startsWith("+") || original.startsWith("00")) {
    const plusified = original.startsWith("00") ? ("+" + original.slice(2)) : original;
    const e164 = toCompactE164(plusified);
    if (!e164) return { e164: null, displayHint: null };

    // If HU → format nicely; else preserve user's original formatting for display
    if (e164.startsWith("+36")) {
      const display = formatHUForDisplay(e164); // defined below
      return { e164, displayHint: display };
    }
    return { e164, displayHint: original }; // preserve non-HU formatting
  }

  // 2) Try Hungarian normalizations for national-style inputs
  // Accept common HU inputs: 06XXXXXXXXX, 36XXXXXXXXX, bare 9 digits, etc.
  let s = original.replace(/\s+/g, "").replace(/[^\d]/g, ""); // remove spaces only for detection
  if (s.startsWith("06")) s = "36" + s.slice(2);
  else if (s.startsWith("36")) s = s;
  else if (/^\d{9}$/.test(s)) s = "36" + s;         // e.g. 701234567
  else if (/^0\d{9}$/.test(s)) s = "36" + s.slice(1); // e.g. 0701234567

  if (/^36\d{9}$/.test(s)) {
    const e164 = "+" + s;
    return { e164, displayHint: formatHUForDisplay(e164) };
  }

  // 3) Anything else (non-HU without explicit country) → invalid (we can’t guess)
  return { e164: null, displayHint: null };
}

/**
 * Pretty format Hungarian E.164 ("+36XXXXXXXXX") for profile display.
 * - Budapest (1): +36 1 234 5678
 * - Others/mobiles (2-digit area): +36 AA BBB CCCC
 */
function formatHUForDisplay(e164) {
  if (!/^\+36\d{9}$/.test(e164 || "")) return null;
  const national = e164.slice(3);
  if (national.startsWith("1")) {
    return `+36 1 ${national.slice(1, 4)} ${national.slice(4)}`;
  }
  const area = national.slice(0, 2);
  const sub = national.slice(2);
  return `+36 ${area} ${sub.slice(0, 3)} ${sub.slice(3)}`;
}

/**
 * Convenience: returns { e164, display }
 * - HU → display is nicely grouped HU
 * - non-HU → display is the original user formatting
 */
function normalizeAndFormat(input) {
  const { e164, displayHint } = normalizePhone(input);
  return { e164, display: displayHint };
}
