<?php
header("Access-Control-Allow-Origin: https://www.homeinfo.hu");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

date_default_timezone_set('Europe/Budapest');

// --- Error Reporting (Development vs Production) ---
// error_reporting(E_ALL); // Development
// ini_set('display_errors', 1); // Development
error_reporting(0); // Production
ini_set('display_errors', 0); // Production

set_time_limit(900); // 15 perc
ini_set('memory_limit', '256M'); // 256 MB memória

// --- Include Shared Config/Functions ---
require_once __DIR__ . '/config.php';

// --- Response Helper ---
function json_response($data, $status_code = 200)
{
    http_response_code($status_code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data);
    exit; // Terminate script after sending response
}

// --- Database Connection ---
function db_connect()
{
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        // Log error securely, don't expose details to client in production
        // error_log("Database Connection Error: " . $conn->connect_error);
        json_response(['error' => 'Database connection failed.'], 500);
    }
    $conn->set_charset("utf8mb4");
    return $conn;
}

// --- API Endpoint Logic ---
$action = $_GET['action'] ?? null;
$method = $_SERVER['REQUEST_METHOD'];
$db = db_connect();

switch ($action) {
    // --- Get Notification Logs ---
    case 'get_logs':
        if ($method !== 'GET') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $result = $db->query("SELECT id, title, description, target_type, target_emails, sent_at, fcm_response_status, success_count, failure_count, scheduled_send_at, parent_log_id FROM push_notification_logs ORDER BY id DESC");
        if (!$result) {
            json_response(['error' => 'Failed to retrieve logs: ' . $db->error], 500);
        }
        $logs = [];
        while ($row = $result->fetch_assoc()) {
            settype($row['id'], 'integer');
            $logs[] = $row;
        }
        $result->free();
        json_response($logs);
        break;

    // --- Delete Notification Log ---
    case 'delete_log':
        if ($method !== 'DELETE') { // Use DELETE method semantically
            json_response(['error' => 'Method not allowed'], 405);
        }
        $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$id) {
            json_response(['error' => 'Invalid or missing log ID'], 400);
        }

        $stmt = $db->prepare("DELETE FROM push_notification_logs WHERE id = ?");
        if (!$stmt) {
            json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
        }
        $stmt->bind_param('i', $id);
        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                json_response(['message' => 'Log entry deleted successfully']);
            } else {
                json_response(['error' => 'Log entry not found'], 404);
            }
        } else {
            json_response(['error' => 'Failed to delete log entry: ' . $stmt->error], 500);
        }
        $stmt->close();
        break;

    // --- Get Delivery Details for a Log (NEW ENDPOINT) ---
    case 'get_delivery_details':
        if ($method !== 'GET') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $log_id = filter_input(INPUT_GET, 'log_id', FILTER_VALIDATE_INT);
        if (!$log_id) {
            json_response(['error' => 'Invalid or missing log_id'], 400);
        }

        $stmt = $db->prepare("SELECT id, target_user_id, target_email, target_token, status, fcm_error_message, attempted_at FROM push_notification_delivery_status WHERE push_notification_log_id = ? ORDER BY attempted_at ASC");
        if (!$stmt) {
            json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
        }
        $stmt->bind_param('i', $log_id);
        if (!$stmt->execute()) {
            json_response(['error' => 'Failed to execute statement: ' . $stmt->error], 500);
        }
        $result = $stmt->get_result();
        $details = [];
        while ($row = $result->fetch_assoc()) {
            $details[] = $row;
        }
        $stmt->close();
        json_response($details);
        break;

    // --- Get Log Status (NEW for polling) ---
    case 'get_log_status':
        if ($method !== 'GET') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $log_id = filter_input(INPUT_GET, 'log_id', FILTER_VALIDATE_INT);
        if (!$log_id) {
            json_response(['error' => 'Invalid or missing log_id'], 400);
        }

        $stmt = $db->prepare("SELECT id, fcm_response_status, success_count, failure_count, sent_at, scheduled_send_at FROM push_notification_logs WHERE id = ?");
        if (!$stmt) {
            json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
        }
        $stmt->bind_param('i', $log_id);
        if (!$stmt->execute()) {
            json_response(['error' => 'Failed to execute statement: ' . $stmt->error], 500);
        }
        $result = $stmt->get_result();
        $logStatus = $result->fetch_assoc();
        $stmt->close();

        if ($logStatus) {
            json_response($logStatus);
        } else {
            json_response(['error' => 'Log entry not found'], 404);
        }
        break;


    // --- Send Quiz Reminder (NEW ACTION) ---
    case 'send_quiz_reminder':
        if ($method !== 'POST') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $input = json_decode(file_get_contents('php://input'), true);

        // --- 1. Validation & Input Prep ---
        $title = trim($input['title'] ?? '');
        $description = trim($input['description'] ?? '');
        $scheduledSendAtInput = $input['scheduled_send_at'] ?? null; // Optional scheduling

        // Validate title & description
        if (empty($title) || mb_strlen($title) > 47) {
            json_response(['error' => 'Title is required and must be max 47 characters'], 400);
            exit;
        }
        if (empty($description) || mb_strlen($description) > 400) {
            json_response(['error' => 'Description is required and must be max 400 characters'], 400);
            exit;
        }

        $scheduledSendAt = null;
        if (!empty($scheduledSendAtInput)) {
            // Attempt to parse the datetime string. Use a specific format for reliability.
            // Assumes input is in 'YYYY-MM-DD HH:MM:SS' format. Adjust format if needed.
            $dateTime = DateTime::createFromFormat('Y-m-d H:i:s', $scheduledSendAtInput); // Or use DateTimeInterface::ATOM for ISO 8601

            if ($dateTime === false || $dateTime->format('Y-m-d H:i:s') !== $scheduledSendAtInput) {
                json_response(['error' => 'Invalid scheduled_send_at format. Use YYYY-MM-DD HH:MM:SS.'], 400);
                exit;
            }
            // Optional: Check if scheduled time is in the past (allow or disallow?)
            $now = new DateTime();
            if ($dateTime < $now) {
                // Option 1: Disallow past dates
                json_response(['error' => 'Scheduled time cannot be in the past.'], 400);
                exit;
                // Option 2: Treat past dates as "send immediately"
                //$scheduledSendAt = null; // Set to null to trigger immediate processing
                // Or set to NOW(): $scheduledSendAt = $now->format('Y-m-d H:i:s');
            } else {
                $scheduledSendAt = $dateTime->format('Y-m-d H:i:s'); // Store in MySQL DATETIME format
            }
        }
        // If $scheduledSendAt remains null, it means send immediately.

        // --- 2. Create Queued Log Entry (without knowing exact users yet) ---
        $logId = null;
        $queueStatus = 'queued';
        $targetTypeSpecific = 'quiz_inactive_recent'; // Special type for the worker
        $initialCount = 0; // Worker will determine actual count

        $db->begin_transaction();
        try {
            // Insert log entry, worker will determine users later
            $sql = "INSERT INTO push_notification_logs
                        (title, description, target_type, target_emails, fcm_response_status,
                         success_count, failure_count, sent_at, scheduled_send_at)
                    VALUES (?, ?, ?, NULL, ?, ?, ?, NOW(), ?)"; // target_emails is NULL

            $stmtLogInitial = $db->prepare($sql);
            if (!$stmtLogInitial) throw new Exception("Prepare failed (queue log): " . $db->error);

            // Note bind types: ss s s ii s (title, desc, targetTypeSpecific, queueStatus, count, count, scheduledSendAt)
            $stmtLogInitial->bind_param(
                'ssssiiss',
                $title,
                $description,
                $targetTypeSpecific,
                $queueStatus,
                $initialCount,
                $initialCount,
                $scheduledSendAt
            );

            if (!$stmtLogInitial->execute()) throw new Exception("Execute failed (queue log): " . $stmtLogInitial->error);

            $logId = $stmtLogInitial->insert_id;
            $stmtLogInitial->close();
            if (!$logId) throw new Exception("Failed to get insert ID for queued log.");

            $db->commit();

            // --- 3. Respond Immediately ---
            json_response([
                'message' => $scheduledSendAt ? "Quiz reminder job scheduled successfully for {$scheduledSendAt}." : 'Quiz reminder job queued successfully for immediate processing.',
                'logId' => $logId,
                'status' => $queueStatus,
                'target_type' => $targetTypeSpecific,
                'scheduled_send_at' => $scheduledSendAt
            ], 202);
        } catch (Exception $e) {
            $db->rollback();
            error_log("Quiz Reminder Queue Error: " . $e->getMessage());
            json_response(['error' => 'Failed to queue quiz reminder job: ' . $e->getMessage()], 500);
        }
        break; // End of 'send_quiz_reminder' case

    // --- Send Notification (MODIFIED TO HANDLE OPTIONAL REMINDER CREATION) ---
    case 'send':
        if ($method !== 'POST') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $input = json_decode(file_get_contents('php://input'), true);

        // --- 1. Validation & Input Prep ---
        $title = trim($input['title'] ?? '');
        $description = trim($input['description'] ?? '');
        $targetType = $input['targetType'] ?? ''; // 'all', 'specific', 'emw_nincs_jegye'
        $targetEmailsInput = $input['targetEmails'] ?? '';
        $scheduledSendAtInput = $input['scheduled_send_at'] ?? null; // e.g., "YYYY-MM-DD HH:MM:SS"
        $createReminder = filter_var($input['create_reminder'] ?? false, FILTER_VALIDATE_BOOLEAN); // New flag

        // Standard Validation
        if (empty($title) || mb_strlen($title) > 47) { /* error */
        }
        if (empty($description) || mb_strlen($description) > 400) { /* error */
        }
        if (!in_array($targetType, ['all', 'specific', 'emw_nincs_jegye'])) { /* error */
        }
        if ($targetType === 'specific' && empty($targetEmailsInput)) { /* error */
        }

        // Validate schedule time (needed if reminder is requested)
        $originalScheduledSendAt = null; // Stores the valid original schedule DateTime object
        $originalScheduledSendAtStr = null; // Stores the string for DB insert
        if (!empty($scheduledSendAtInput)) {
            $dateTime = DateTime::createFromFormat('Y-m-d H:i:s', $scheduledSendAtInput);
            if ($dateTime === false || $dateTime->format('Y-m-d H:i:s') !== $scheduledSendAtInput) {
                json_response(['error' => 'Invalid scheduled_send_at format. Use YYYY-MM-DD HH:MM:SS.'], 400);
                exit;
            }
            $now = new DateTime();
            if ($dateTime < $now) {
                // Treat past dates as "send immediately" for the original
                $originalScheduledSendAt = null; // Process immediately
                $originalScheduledSendAtStr = null;
                // If reminder requested for past date, schedule reminder 1 day from *now*? Or disallow? Let's schedule from now.
                if ($createReminder) {
                    echo "Warning: Original schedule time is in the past. Reminder will be scheduled 1 day from now.\n";
                }
            } else {
                $originalScheduledSendAt = $dateTime; // Keep DateTime object for calculation
                $originalScheduledSendAtStr = $dateTime->format('Y-m-d H:i:s');
            }
        } else {
            // Immediate send for original
            $originalScheduledSendAt = null;
            $originalScheduledSendAtStr = null;
        }

        // Crucial Check: Reminder requires an original schedule time (or we base it on 'now')
        if ($createReminder && !$originalScheduledSendAt && !$originalScheduledSendAtStr && empty($scheduledSendAtInput)) {
            // If original send is immediate, base reminder on 'now' + 1 day
            $originalScheduledSendAt = new DateTime(); // Use current time as base for reminder calc
        } else if ($createReminder && !$originalScheduledSendAt) {
            // If original schedule was in the past, base reminder on 'now'
            $originalScheduledSendAt = new DateTime();
        } else if ($createReminder && !$originalScheduledSendAt && !empty($scheduledSendAtInput)) {
            // This case should not happen if validation passed, but as fallback:
            json_response(['error' => 'Cannot create reminder without a valid future schedule time for the original notification.'], 400);
            exit;
        }


        $targetEmailsLog = ($targetType === 'specific') ? $targetEmailsInput : null;

        // --- 2. Create Log Entries (Original + Optional Reminder) ---
        $originalLogId = null;
        $reminderLogId = null;
        $queueStatus = 'queued';
        $initialCount = 0;
        $reminderScheduledAtStr = null; // For response message

        $db->begin_transaction();
        try {
            // --- Insert Original Log Entry ---
            $sqlOriginal = "INSERT INTO push_notification_logs
                        (title, description, target_type, target_emails, fcm_response_status,
                         success_count, failure_count, sent_at, scheduled_send_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), ?)";
            $stmtOriginal = $db->prepare($sqlOriginal);
            if (!$stmtOriginal) throw new Exception("Prepare failed (original log): " . $db->error);
            $stmtOriginal->bind_param(
                'sssssiis',
                $title,
                $description,
                $targetType,
                $targetEmailsLog,
                $queueStatus,
                $initialCount,
                $initialCount,
                $originalScheduledSendAtStr
            );
            if (!$stmtOriginal->execute()) throw new Exception("Execute failed (original log): " . $stmtOriginal->error);
            $originalLogId = $stmtOriginal->insert_id;
            $stmtOriginal->close();
            if (!$originalLogId) throw new Exception("Failed to get insert ID for original log.");

            // --- Insert Reminder Log Entry (if requested) ---
            if ($createReminder && $originalLogId && $originalScheduledSendAt) {
                $reminderVariations = [
                    ['Ne felejtsd el: a kvíz már csak rád vár!', 'Mutasd meg, mit tudsz!'],
                    ['Még nem késő beszállni!', 'Töltsd ki a kvízt, és nézd meg, hány pontot érsz el!'],
                    ['Ki töltötted már a tegnapi kvízt?', 'A kérdések csak rád várnak!'],
                    ['A kvíz csak rád vár vár!', 'Siess, hogy megszerezd a legnagyobb pontokat!'],
                    ['Kvíz emlékeztető!', 'Töltsd ki a kvízt, és nézd meg, hány pontot érsz el!']
                ];
                $rnd = mt_rand(0, count($reminderVariations) - 1); // Randomize reminder variation              
                $reminderTitle = $reminderVariations[$rnd][0]; // Default reminder title
                $reminderDesc = $reminderVariations[$rnd][1]; // Default reminder description
                $reminderTargetType = 'quiz_inactive_recent'; // Specific target type for reminder

                // Calculate reminder schedule time (+1 day from original schedule base)
                $reminderScheduledAt = clone $originalScheduledSendAt; // Clone to avoid modifying original
                $reminderScheduledAt->add(new DateInterval('P1D')); // Add 1 day
                $reminderScheduledAtStr = $reminderScheduledAt->format('Y-m-d H:i:s');

                $sqlReminder = "INSERT INTO push_notification_logs
                            (parent_log_id, title, description, target_type, target_emails, fcm_response_status,
                             success_count, failure_count, sent_at, scheduled_send_at)
                        VALUES (?, ?, ?, ?, NULL, ?, ?, ?, NOW(), ?)"; // parent_log_id added

                $stmtReminder = $db->prepare($sqlReminder);
                if (!$stmtReminder) throw new Exception("Prepare failed (reminder log): " . $db->error);
                // Note bind types: i sss si i s (9 params)
                $stmtReminder->bind_param(
                    'issssiis',
                    $originalLogId, // Link to parent
                    $reminderTitle,
                    $reminderDesc,
                    $reminderTargetType,
                    $queueStatus,
                    $initialCount,
                    $initialCount,
                    $reminderScheduledAtStr
                );
                if (!$stmtReminder->execute()) throw new Exception("Execute failed (reminder log): " . $stmtReminder->error);
                $reminderLogId = $stmtReminder->insert_id;
                $stmtReminder->close();
                if (!$reminderLogId) throw new Exception("Failed to get insert ID for reminder log.");
            }

            $db->commit(); // Commit both inserts

            // --- 3. Respond Immediately ---
            $responseMessage = $originalScheduledSendAtStr
                ? "Original notification job scheduled successfully for {$originalScheduledSendAtStr}."
                : 'Original notification job queued successfully for immediate processing.';
            if ($reminderLogId && $reminderScheduledAtStr) {
                $responseMessage .= " Automatic reminder scheduled for {$reminderScheduledAtStr}.";
            }

            json_response([
                'message' => $responseMessage,
                'originalLogId' => $originalLogId,
                'reminderLogId' => $reminderLogId, // Include if created
                'status' => $queueStatus,
                'original_scheduled_send_at' => $originalScheduledSendAtStr,
                'reminder_scheduled_send_at' => $reminderScheduledAtStr // Include if created
            ], 202); // 202 Accepted

        } catch (Exception $e) {
            $db->rollback(); // Rollback on any error
            error_log("Notification Queue Error (with Reminder): " . $e->getMessage());
            json_response(['error' => 'Failed to queue notification job: ' . $e->getMessage()], 500);
        }
        break; // End of 'send' case

    // --- Default: Action not found ---
    default:
        json_response(['error' => 'Invalid action specified'], 400);
        break;
}

// Close DB connection
$db->close();
