<?php
header("Access-Control-Allow-Origin: https://www.homeinfo.hu");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

require_once "../vendor/autoload.php";

$credential = new Google\Auth\Credentials\ServiceAccountCredentials(
    "https://www.googleapis.com/auth/firebase.messaging",
    json_decode(file_get_contents('https://www.homeinfo.hu/vip/pvKey.json'), TRUE)
);
$authToken = $credential->fetchAuthToken(Google\Auth\HttpHandler\HttpHandlerFactory::build());

// --- Configuration ---
// !! IMPORTANT !! Replace with your actual credentials
// !! Consider using environment variables or a config file outside web root for security !!
define('DB_HOST', 'localhost'); // Vagy az adatbázis szerver címe
define('DB_NAME', 'dasl1_db_inspiracio'); // Cseréld le az adatbázis nevére
define('DB_USER', 'dasl1_usr_inspir'); // Cseréld le a felhasználónévre
define('DB_PASS', 'p-^u;.TdrB*5'); // Cseréld le a jelszóra

// Firebase Cloud Messaging (FCM) v1 API configuration
define('FCM_PROJECT_ID', 'homeinfo-vip'); // Your Firebase Project ID
define('FCM_API_URL', 'https://fcm.googleapis.com/v1/projects/' . FCM_PROJECT_ID . '/messages:send');
define('FCM_SERVER_KEY', $authToken['access_token']); // Replace with your actual FCM Server Key

// --- Error Reporting (Development vs Production) ---
// error_reporting(E_ALL); // Development
// ini_set('display_errors', 1); // Development
error_reporting(0); // Production
ini_set('display_errors', 0); // Production

set_time_limit(900); // 15 perc
ini_set('memory_limit', '256M'); // 256 MB memória

// --- Response Helper ---
function json_response($data, $status_code = 200)
{
    http_response_code($status_code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data);
    exit; // Terminate script after sending response
}

// --- Database Connection ---
function db_connect()
{
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        // Log error securely, don't expose details to client in production
        // error_log("Database Connection Error: " . $conn->connect_error);
        json_response(['error' => 'Database connection failed.'], 500);
    }
    $conn->set_charset("utf8mb4");
    return $conn;
}

// --- FCM Sending Function ---
// Note: FCM v1 API sends one message per request. We loop through tokens.
// For large numbers of tokens, consider batching or alternative strategies (e.g., topics).
function send_fcm_notification($token, $title, $description)
{
    $payload = [
        'message' => [
            'token' => $token,
            'notification' => [
                'title' => $title,
                'body' => $description,
            ],
            // Optional: Add 'data' payload if needed by your app
            // 'data' => [ 'key' => 'value' ]
        ]
    ];

    $headers = [
        'Authorization: Bearer ' . FCM_SERVER_KEY, // Or use OAuth2 token if using Service Account
        'Content-Type: application/json'
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, FCM_API_URL);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Set to true in production with proper CA certs
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

    $result = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($curl_error) {
        return ['success' => false, 'error' => 'Curl error: ' . $curl_error, 'response' => null, 'http_code' => $http_code];
    }

    $response_data = json_decode($result, true);

    // FCM v1 returns 200 OK even for errors like unregistered tokens, check response body
    if ($http_code === 200 && isset($response_data['name'])) { // 'name' field indicates success
        return ['success' => true, 'error' => null, 'response' => $response_data, 'http_code' => $http_code];
    } else {
        // Attempt to extract a more specific error message if available
        $error_message = 'FCM API error';
        if (isset($response_data['error']['message'])) {
            $error_message = $response_data['error']['message'];
        } elseif ($result) {
            $error_message = $result; // Use raw response if JSON parsing failed or structure unknown
        }
        return ['success' => false, 'error' => $error_message, 'response' => $response_data, 'http_code' => $http_code];
    }
}

// --- Function to handle unsubscription ---
function unsubscribe_fcm_token(mysqli $db, int $userId, string $token)
{
    // Option A: Nullify the token (Recommended)
    $stmt = $db->prepare("UPDATE inspiracio_fbuser SET token = NULL WHERE id = ? AND token = ?");
    // Using both id and token adds a layer of safety ensuring you nullify the correct token for that user
    if ($stmt) {
        $stmt->bind_param('is', $userId, $token);
        if (!$stmt->execute()) {
            error_log("Failed to nullify token for user ID {$userId}: " . $stmt->error);
            // Handle error appropriately - maybe just log it
        } else {
            error_log("Successfully nullified token for user ID {$userId}"); // Debug logging
        }
        $stmt->close();
    } else {
        error_log("Failed to prepare statement to nullify token for user ID {$userId}: " . $db->error);
    }
}

// --- Function to Fetch Target Users (Helper to avoid code duplication) ---
function get_target_users(mysqli $db, string $targetType, ?string $targetEmailsInput): array
{
    $targetUsers = []; // Will store {id, mail, token}

    if ($targetType === 'all') {
        $result = $db->query("SELECT id, mail, token FROM inspiracio_fbuser WHERE token IS NOT NULL AND token != '' AND mester_app_login > 0 AND unregistered = 0");
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $targetUsers[] = $row;
            }
            $result->free();
        }
    } elseif ($targetType === 'specific') {
        if (empty($targetEmailsInput)) {
            throw new Exception("Target emails are required for specific targeting."); // Or return empty array/handle differently
        }
        $targetEmailsArray = array_map('trim', explode(',', $targetEmailsInput));
        // $targetEmailsArray = array_filter($targetEmailsArray, fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));
        $targetEmailsArray = array_filter($targetEmailsArray, function ($email) {
            return filter_var($email, FILTER_VALIDATE_EMAIL);
        });

        if (!empty($targetEmailsArray)) {
            $placeholders = implode(',', array_fill(0, count($targetEmailsArray), '?'));
            $types = str_repeat('s', count($targetEmailsArray));
            $sql = "SELECT id, mail, token FROM inspiracio_fbuser WHERE mail IN ($placeholders) AND token IS NOT NULL AND token != ''";

            $stmt = $db->prepare($sql);
            if (!$stmt) {
                throw new Exception("Prepare statement failed (get specific users): " . $db->error);
            }
            $stmt->bind_param($types, ...$targetEmailsArray);
            if (!$stmt->execute()) {
                throw new Exception("Execute failed (get specific users): " . $stmt->error);
            }
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $targetUsers[] = $row;
            }
            $stmt->close();
        }
    }
    return $targetUsers;
}

// --- API Endpoint Logic ---
$action = $_GET['action'] ?? null;
$method = $_SERVER['REQUEST_METHOD'];
$db = db_connect();

switch ($action) {
    // --- Get Notification Logs ---
    case 'get_logs':
        if ($method !== 'GET') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $result = $db->query("SELECT id, title, description, target_type, target_emails, sent_at, fcm_response_status, success_count, failure_count FROM push_notification_logs ORDER BY sent_at DESC");
        if (!$result) {
            json_response(['error' => 'Failed to retrieve logs: ' . $db->error], 500);
        }
        $logs = [];
        while ($row = $result->fetch_assoc()) {
            settype($row['id'], 'integer');
            $logs[] = $row;
        }
        $result->free();
        json_response($logs);
        break;

    // --- Delete Notification Log ---
    case 'delete_log':
        if ($method !== 'DELETE') { // Use DELETE method semantically
            json_response(['error' => 'Method not allowed'], 405);
        }
        $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$id) {
            json_response(['error' => 'Invalid or missing log ID'], 400);
        }

        $stmt = $db->prepare("DELETE FROM push_notification_logs WHERE id = ?");
        if (!$stmt) {
            json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
        }
        $stmt->bind_param('i', $id);
        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                json_response(['message' => 'Log entry deleted successfully']);
            } else {
                json_response(['error' => 'Log entry not found'], 404);
            }
        } else {
            json_response(['error' => 'Failed to delete log entry: ' . $stmt->error], 500);
        }
        $stmt->close();
        break;

    // --- Get Delivery Details for a Log (NEW ENDPOINT) ---
    case 'get_delivery_details':
        if ($method !== 'GET') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $log_id = filter_input(INPUT_GET, 'log_id', FILTER_VALIDATE_INT);
        if (!$log_id) {
            json_response(['error' => 'Invalid or missing log_id'], 400);
        }

        $stmt = $db->prepare("SELECT id, target_user_id, target_email, target_token, status, fcm_error_message, attempted_at FROM push_notification_delivery_status WHERE push_notification_log_id = ? ORDER BY attempted_at ASC");
        if (!$stmt) {
            json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
        }
        $stmt->bind_param('i', $log_id);
        if (!$stmt->execute()) {
            json_response(['error' => 'Failed to execute statement: ' . $stmt->error], 500);
        }
        $result = $stmt->get_result();
        $details = [];
        while ($row = $result->fetch_assoc()) {
            $details[] = $row;
        }
        $stmt->close();
        json_response($details);
        break;

    // --- Send Notification ---
    case 'send':
        if ($method !== 'POST') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $input = json_decode(file_get_contents('php://input'), true);

        // Basic Validation
        $title = trim($input['title'] ?? '');
        $description = trim($input['description'] ?? '');
        $targetType = $input['targetType'] ?? ''; // 'all' or 'specific'
        $targetEmailsInput = $input['targetEmails'] ?? ''; // Comma-separated string

        if (empty($title) || mb_strlen($title) > 47) {
            json_response(['error' => 'Title is required and must be max 47 characters'], 400);
        }
        if (empty($description) || mb_strlen($description) > 400) {
            json_response(['error' => 'Description is required and must be max 400 characters'], 400);
        }
        if (!in_array($targetType, ['all', 'specific'])) {
            json_response(['error' => 'Invalid target type specified'], 400);
        }

        $targetUsers = []; // Will store {id, mail, token}
        $targetEmailsLog = null;

        // Fetch Target Tokens
        if ($targetType === 'all') {
            /*$result = $db->query("SELECT id, mail, token FROM inspiracio_fbuser WHERE token IS NOT NULL AND token != '' AND mester_app_login > 0 AND unregistered = 0 limit 1000, 5000;");
            while ($row = $result->fetch_assoc()) {
                $targetUsers[] = $row;
            }
            $result->free();*/
            $targetUsers = get_target_users($db, 'all', $targetEmailsInput);
        } else { // 'specific'
            /* if (empty($targetEmailsInput)) {
                json_response(['error' => 'Target emails are required for specific targeting'], 400);
            }
            // Basic email list cleaning
            $targetEmailsArray = array_map('trim', explode(',', $targetEmailsInput));
            // $targetEmailsArray = array_filter($targetEmailsArray, fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));
            $targetEmailsArray = array_filter($targetEmailsArray, function ($email) {
                return filter_var($email, FILTER_VALIDATE_EMAIL);
            });

            if (empty($targetEmailsArray)) {
                json_response(['error' => 'No valid target emails provided'], 400);
            }

            $placeholders = implode(',', array_fill(0, count($targetEmailsArray), '?'));
            $types = str_repeat('s', count($targetEmailsArray));
            $sql = "SELECT id, mail, token FROM inspiracio_fbuser WHERE mail IN ($placeholders) AND token IS NOT NULL AND token != ''";

            $stmt = $db->prepare($sql);
            if (!$stmt) {
                json_response(['error' => 'Prepare statement failed: ' . $db->error], 500);
            }
            $stmt->bind_param($types, ...$targetEmailsArray);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $targetUsers[] = $row;
            }
            $stmt->close();
            
            $targetEmailsLog = implode(',', $targetEmailsArray); // Store the validated list for logging
            */
            $targetUsers = get_target_users($db, 'specific', $targetEmailsInput);
            $targetEmailsLog = implode(',', array_column($targetUsers, 'mail')); // Store the validated list for logging
        }

        if (empty($targetUsers)) {
            json_response(['error' => 'No valid users/tokens found for the specified targets.'], 404);
        }
        $totalTargets = count($targetUsers);

        // --- Start Transaction ---
        $db->begin_transaction();

        $logId = null;
        $initialLogSuccess = false;
        try {
            // --- 3. Create Initial Main Log Entry ---
            $initialStatus = 'processing'; // Mark as processing initially
            $initialSuccessCount = 0;
            $initialFailureCount = 0;

            $stmtLogInitial = $db->prepare("INSERT INTO push_notification_logs (title, description, target_type, target_emails, fcm_response_status, success_count, failure_count) VALUES (?, ?, ?, ?, ?, ?, ?)");
            if (!$stmtLogInitial) {
                throw new Exception("Prepare failed (initial log): " . $db->error);
            }
            $stmtLogInitial->bind_param('sssssii', $title, $description, $targetType, $targetEmailsLog, $initialStatus, $initialSuccessCount, $initialFailureCount);
            if (!$stmtLogInitial->execute()) {
                throw new Exception("Execute failed (initial log): " . $stmtLogInitial->error);
            }
            $logId = $stmtLogInitial->insert_id;
            $stmtLogInitial->close();
            $initialLogSuccess = true; // Mark that the initial log was created

            if (!$logId) {
                throw new Exception("Failed to get insert ID for initial log.");
            }

            // --- 4. Process Each User (Send & Log Detail Immediately) ---
            $currentSuccessCount = 0;
            $currentFailureCount = 0;
            $currentUnsubscribedCount = 0;

            // Prepare the detail insert statement *once* outside the loop for efficiency
            $stmtDetail = $db->prepare("INSERT INTO push_notification_delivery_status (push_notification_log_id, target_user_id, target_email, target_token, status, fcm_error_message) VALUES (?, ?, ?, ?, ?, ?)");
            if (!$stmtDetail) {
                throw new Exception("Prepare failed (detail log): " . $db->error); // Critical failure if can't prepare
            }

            foreach ($targetUsers as $user) {
                // Set defaults for this iteration
                $attemptStatus = 'failure';
                $attemptErrorMessage = 'Processing skipped or failed before FCM call';
                $shouldUnsubscribe = false;

                try {
                    if (empty($user['token'])) {
                        $attemptErrorMessage = 'User token was missing in database';
                        // $currentFailureCount++; // Already accounted for below generally
                        // Skip FCM call, proceed to log failure detail
                    } else {
                        // --- Send Actual Notification ---
                        $fcmResult = send_fcm_notification($user['token'], $title, $description);
                        $isSuccess = $fcmResult['success'];
                        $attemptErrorMessage = $fcmResult['error'] ?? null; // Error or null if success

                        if ($isSuccess) {
                            $attemptStatus = 'success';
                        } else {
                            $attemptStatus = 'failure';
                            // --- Check for Unsubscription ---
                            $fcmErrorString = $attemptErrorMessage ?: '';
                            $fcmResponseData = $fcmResult['response'] ?? [];
                            $fcmErrorStatus = $fcmResponseData['error']['status'] ?? null;

                            if (
                                $fcmErrorStatus === 'UNREGISTERED' ||
                                $fcmErrorStatus === 'INVALID_ARGUMENT' || // Cautious use
                                $fcmErrorStatus === 'NOT_FOUND' ||
                                (!$fcmErrorStatus && (
                                    strpos($fcmErrorString, 'Unregistered') !== false ||
                                    strpos($fcmErrorString, 'invalid-registration-token') !== false ||
                                    strpos($fcmErrorString, 'registration-token-not-registered') !== false ||
                                    strpos($fcmErrorString, 'InvalidRegistration') !== false
                                ))
                            ) {
                                $shouldUnsubscribe = true;
                                $attemptErrorMessage .= ' (Token Marked Invalid)';
                                // $currentUnsubscribedCount++; // Counted below after successful unsubscribe call
                            }
                        }
                    } // End if/else token check

                } catch (Exception $e) {
                    // Catch errors during the FCM call itself or analysis
                    $attemptStatus = 'failure';
                    $attemptErrorMessage = "Error during processing token: " . $e->getMessage();
                    error_log("Error processing user ID {$user['id']} / token {$user['token']}: " . $e->getMessage());
                }

                // --- Update Counts ---
                if ($attemptStatus === 'success') {
                    $currentSuccessCount++;
                } else {
                    $currentFailureCount++;
                }

                // --- Perform Unsubscription if needed ---
                if ($shouldUnsubscribe) {
                    // Wrap the unsubscribe call in its own try/catch if needed,
                    // so its failure doesn't prevent logging the detail record
                    try {
                        unsubscribe_fcm_token($db, $user['id'], $user['token']);
                        $currentUnsubscribedCount++; // Only increment if unsubscribe was attempted/successful
                    } catch (Exception $unsubError) {
                        error_log("Error during unsubscribe for user ID {$user['id']}: " . $unsubError->getMessage());
                        // Decide if $attemptErrorMessage should be updated here
                    }
                }

                // --- Log Detail Record ---
                $userId = $user['id'];
                $email = $user['mail'];
                $token = $user['token'] ?? '(missing)'; // Use actual token or placeholder

                // Use the prepared statement $stmtDetail
                $stmtDetail->bind_param('iissss', $logId, $userId, $email, $token, $attemptStatus, $attemptErrorMessage);
                if (!$stmtDetail->execute()) {
                    // Log this specific detail error but *continue the loop*
                    error_log("Failed to log detail for user {$userId} / token {$token} (Log ID: {$logId}): " . $stmtDetail->error);
                    // Don't throw; allow the batch to continue
                    // Maybe decrement totalTargets if you want final status to reflect only successfully logged details? complex.
                }

                // Optional delay
                // usleep(10000);

            } // --- End foreach loop ---

            $stmtDetail->close(); // Close the prepared statement after the loop

            // --- 5. Update Main Log Entry with Final Counts & Status ---
            $finalStatus = 'failure'; // Default
            if ($currentSuccessCount > 0 && $currentFailureCount == 0) {
                $finalStatus = 'success';
            } elseif ($currentSuccessCount > 0 && $currentFailureCount > 0) {
                $finalStatus = 'partial_failure';
            } elseif ($currentSuccessCount == 0 && $currentFailureCount > 0) {
                $finalStatus = 'failure';
            } else { // E.g., 0 success, 0 failure (maybe all targets were invalid before sending?)
                if ($totalTargets > 0) {
                    $finalStatus = 'failure'; // Treat as failure if targets existed but none were processed
                } else {
                    $finalStatus = 'no_targets'; // Or 'success' if 0 targets is considered success
                }
            }

            $stmtLogUpdate = $db->prepare("UPDATE push_notification_logs SET success_count = ?, failure_count = ?, fcm_response_status = ? WHERE id = ?");
            if (!$stmtLogUpdate) {
                // This is a more critical failure after processing
                throw new Exception("Prepare failed (update log): " . $db->error);
            }
            $stmtLogUpdate->bind_param('iisi', $currentSuccessCount, $currentFailureCount, $finalStatus, $logId);
            if (!$stmtLogUpdate->execute()) {
                throw new Exception("Execute failed (update log): " . $stmtLogUpdate->error);
            }
            $stmtLogUpdate->close();

            // --- 6. Commit Transaction ---
            $db->commit();

            // --- 7. Respond to Client ---
            json_response([
                'message' => 'Notification request processed.',
                'logId' => $logId,
                'status' => $finalStatus, // Use the final calculated status
                'successCount' => $currentSuccessCount,
                'failureCount' => $currentFailureCount,
                'unsubscribedCount' => $currentUnsubscribedCount
            ]);
        } catch (Exception $e) {
            // --- Rollback Transaction on Error ---
            if ($db->in_transaction) { // Check if transaction was started
                $db->rollback();
            }

            error_log("Notification Send/Log Error (Log ID attempt: {$logId}): " . $e->getMessage());

            // If the initial log failed, $logId might be null
            $errorResponse = [
                'message' => 'Notification processing failed.',
                'error' => $e->getMessage(), // Be cautious exposing raw errors
                'logId' => $logId // Include logId if initial log was created
            ];
            // If the initial log *was* created, maybe update its status to 'failed' if possible?
            if ($initialLogSuccess && $logId) {
                try {
                    $stmtFail = $db->prepare("UPDATE push_notification_logs SET fcm_response_status = 'failed' WHERE id = ?");
                    if ($stmtFail) {
                        $stmtFail->bind_param('i', $logId);
                        $stmtFail->execute();
                        $stmtFail->close();
                    }
                } catch (Exception $updateFailError) {
                    error_log("Failed to mark log {$logId} as failed after rollback: " . $updateFailError->getMessage());
                }
                $errorResponse['message'] .= " Log entry {$logId} may be incomplete.";
            }

            json_response($errorResponse, 500); // Internal Server Error
        }

        break; // End of 'send' case
    // --- Resume Send (NEW ACTION) ---
    case 'resume_send':
        if ($method !== 'POST') {
            json_response(['error' => 'Method not allowed'], 405);
        }
        $input = json_decode(file_get_contents('php://input'), true);
        $logId = filter_var($input['log_id'] ?? null, FILTER_VALIDATE_INT);

        if (!$logId) {
            json_response(['error' => 'Invalid or missing log_id'], 400);
        }

        // --- Start Transaction ---
        $db->begin_transaction();
        $originalLogData = null;

        try {
            // --- 1. Fetch Original Log & Verify Status ---
            $stmtGetLog = $db->prepare("SELECT id, title, description, target_type, target_emails, success_count, failure_count, fcm_response_status FROM push_notification_logs WHERE id = ? FOR UPDATE");
            // "FOR UPDATE" locks the row to prevent concurrent resume attempts
            if (!$stmtGetLog) throw new Exception("Prepare failed (get log): " . $db->error);
            $stmtGetLog->bind_param('i', $logId);
            if (!$stmtGetLog->execute()) throw new Exception("Execute failed (get log): " . $stmtGetLog->error);
            $resultLog = $stmtGetLog->get_result();
            $originalLogData = $resultLog->fetch_assoc();
            $stmtGetLog->close();

            if (!$originalLogData) {
                throw new Exception("Log entry not found.");
            }
            if ($originalLogData['fcm_response_status'] !== 'processing') {
                throw new Exception("This job is not in a 'processing' state and cannot be resumed.");
            }

            // Extract necessary data
            $title = $originalLogData['title'];
            $description = $originalLogData['description'];
            $targetType = $originalLogData['target_type'];
            $targetEmailsInput = $originalLogData['target_emails']; // Might be NULL
            $originalSuccessCount = (int)$originalLogData['success_count'];
            $originalFailureCount = (int)$originalLogData['failure_count'];

            // --- 2. Determine ALL Original Target Users ---
            // Use the helper function
            $allOriginalTargetUsers = get_target_users($db, $targetType, $targetEmailsInput);
            $totalOriginalTargets = count($allOriginalTargetUsers);


            // --- 3. Identify Already Processed Users ---
            $processedUserIds = []; // Store IDs of users already processed
            $stmtGetProcessed = $db->prepare("SELECT target_user_id FROM push_notification_delivery_status WHERE push_notification_log_id = ? AND target_user_id IS NOT NULL");
            if (!$stmtGetProcessed) throw new Exception("Prepare failed (get processed): " . $db->error);
            $stmtGetProcessed->bind_param('i', $logId);
            if (!$stmtGetProcessed->execute()) throw new Exception("Execute failed (get processed): " . $stmtGetProcessed->error);
            $resultProcessed = $stmtGetProcessed->get_result();
            while ($row = $resultProcessed->fetch_assoc()) {
                $processedUserIds[$row['target_user_id']] = true; // Use user ID as key for quick lookup
            }
            $stmtGetProcessed->close();
            // Note: If target_user_id can be NULL, you might need to query/compare by token instead, which is less efficient.


            // --- 4. Identify Users Still Needing Processing ---
            $usersToProcess = [];
            foreach ($allOriginalTargetUsers as $user) {
                // Check if user ID exists and is NOT in the processed list
                if (isset($user['id']) && !isset($processedUserIds[$user['id']])) {
                    $usersToProcess[] = $user;
                }
                // Add handling here if you need to fallback to token comparison when ID is missing
            }

            // --- 5. Process Remaining Users ---
            $newSuccessCount = 0;
            $newFailureCount = 0;
            $newUnsubscribedCount = 0;

            if (!empty($usersToProcess)) {
                // Prepare the detail insert statement *once*
                $stmtDetail = $db->prepare("INSERT INTO push_notification_delivery_status (push_notification_log_id, target_user_id, target_email, target_token, status, fcm_error_message) VALUES (?, ?, ?, ?, ?, ?)");
                if (!$stmtDetail) throw new Exception("Prepare failed (detail log): " . $db->error);

                foreach ($usersToProcess as $user) {
                    // --- Copy & adapt the inner loop logic from 'send' action ---
                    $attemptStatus = 'failure';
                    $attemptErrorMessage = 'Processing skipped or failed before FCM call';
                    $shouldUnsubscribe = false;

                    try {
                        if (empty($user['token'])) {
                            $attemptErrorMessage = 'User token was missing in database';
                        } else {
                            // Send Actual Notification
                            $fcmResult = send_fcm_notification($user['token'], $title, $description);
                            // Analyze result, Check for Unsubscription
                            // ... (exact same logic as in the 'send' action's loop) ...
                            $isSuccess = $fcmResult['success'];
                            $attemptErrorMessage = $fcmResult['error'] ?? null;
                            if ($isSuccess) {
                                $attemptStatus = 'success';
                            } else {
                                // Check for unsubscribe conditions...
                                $fcmErrorString = $attemptErrorMessage ?: '';
                                $fcmResponseData = $fcmResult['response'] ?? [];
                                $fcmErrorStatus = $fcmResponseData['error']['status'] ?? null;
                                if (/* condition matches unsubscribe */
                                    $fcmErrorStatus === 'UNREGISTERED' || /* ... other conditions ... */ false
                                ) {
                                    $shouldUnsubscribe = true;
                                    $attemptErrorMessage .= ' (Token Marked Invalid)';
                                }
                            }
                        }
                    } catch (Exception $e) {
                        $attemptStatus = 'failure';
                        $attemptErrorMessage = "Error during processing token: " . $e->getMessage();
                        error_log("Resume Error processing user ID {$user['id']} / token {$user['token']} for Log ID {$logId}: " . $e->getMessage());
                    }

                    // Update Counts for *this session*
                    if ($attemptStatus === 'success') $newSuccessCount++;
                    else $newFailureCount++;

                    // Perform Unsubscription
                    if ($shouldUnsubscribe) {
                        try {
                            unsubscribe_fcm_token($db, $user['id'], $user['token']);
                            $newUnsubscribedCount++;
                        } catch (Exception $unsubError) { /* log error */
                        }
                    }

                    // Log Detail Record
                    $userId = $user['id'];
                    $email = $user['mail'];
                    $token = $user['token'] ?? '(missing)';
                    $stmtDetail->bind_param('iissss', $logId, $userId, $email, $token, $attemptStatus, $attemptErrorMessage);
                    if (!$stmtDetail->execute()) {
                        error_log("Resume: Failed to log detail for user {$userId} / token {$token} (Log ID: {$logId}): " . $stmtDetail->error);
                        // Continue loop
                    }
                } // --- End foreach loop for usersToProcess ---
                $stmtDetail->close();
            } // --- End if !empty($usersToProcess) ---


            // --- 6. Calculate Final Combined Counts & Status ---
            $finalSuccessCount = $originalSuccessCount + $newSuccessCount;
            $finalFailureCount = $originalFailureCount + $newFailureCount;

            $finalStatus = 'failure'; // Default
            // Use the same logic as in 'send' to determine final status based on FINAL counts
            if ($finalSuccessCount > 0 && $finalFailureCount == 0) $finalStatus = 'success';
            elseif ($finalSuccessCount > 0 && $finalFailureCount > 0) $finalStatus = 'partial_failure';
            elseif ($finalSuccessCount == 0 && $finalFailureCount > 0) $finalStatus = 'failure';
            else { // Handle 0/0 cases
                if ($totalOriginalTargets > 0) $finalStatus = 'failure'; // If targets existed but all failed eventually
                else $finalStatus = 'no_targets'; // Should not happen if initial send started
            }

            // --- 7. Update Main Log Entry ---
            $stmtLogUpdate = $db->prepare("UPDATE push_notification_logs SET success_count = ?, failure_count = ?, fcm_response_status = ? WHERE id = ?");
            if (!$stmtLogUpdate) throw new Exception("Prepare failed (update log): " . $db->error);
            $stmtLogUpdate->bind_param('iisi', $finalSuccessCount, $finalFailureCount, $finalStatus, $logId);
            if (!$stmtLogUpdate->execute()) throw new Exception("Execute failed (update log): " . $stmtLogUpdate->error);
            $stmtLogUpdate->close();

            // --- 8. Commit Transaction ---
            $db->commit();

            // --- 9. Respond to Client ---
            json_response([
                'message' => 'Notification resume processed.',
                'logId' => $logId,
                'status' => $finalStatus,
                'totalSuccessCount' => $finalSuccessCount, // Report final total
                'totalFailureCount' => $finalFailureCount, // Report final total
                'resumedSuccess' => $newSuccessCount,     // Info about this session
                'resumedFailure' => $newFailureCount,     // Info about this session
                'resumedUnsubscribed' => $newUnsubscribedCount // Info about this session
            ]);
        } catch (Exception $e) {
            // --- Rollback Transaction on Error ---
            if ($db->in_transaction) {
                $db->rollback();
            }
            error_log("Notification Resume Error (Log ID: {$logId}): " . $e->getMessage());

            // Attempt to mark the log as 'failed' if it was originally 'processing' and we got this far
            if ($logId && $originalLogData && $originalLogData['fcm_response_status'] === 'processing') {
                try {
                    $stmtFail = $db->prepare("UPDATE push_notification_logs SET fcm_response_status = 'failed' WHERE id = ? AND fcm_response_status = 'processing'");
                    if ($stmtFail) {
                        $stmtFail->bind_param('i', $logId);
                        $stmtFail->execute();
                        $stmtFail->close();
                    }
                } catch (Exception $updateFailError) { /* log error */
                }
            }

            json_response(['error' => "Resume failed: " . $e->getMessage()], 500);
        }

        break; // End of 'resume_send' case

    // --- Default: Action not found ---
    default:
        json_response(['error' => 'Invalid action specified'], 400);
        break;
}

// Close DB connection
$db->close();
