const API_URL = 'https://www.homeinfo.hu/vip/push/api.php'; // Path to your PHP API file

// --- DOM Elements ---
const notificationForm = document.getElementById('notificationForm');
const titleInput = document.getElementById('title');
const descriptionInput = document.getElementById('description');
const targetTypeRadios = document.querySelectorAll('input[name="targetType"]');
const targetEmailsGroup = document.getElementById('targetEmailsGroup');
const targetEmailsInput = document.getElementById('targetEmails');
const sendButton = document.getElementById('sendButton');
const clearFormButton = document.getElementById('clearFormButton');
const formStatus = document.getElementById('formStatus');

const logLoading = document.getElementById('logLoading');
const logError = document.getElementById('logError');
const logTable = document.getElementById('logTable');
const logTableBody = document.getElementById('logTableBody');
const noLogs = document.getElementById('noLogs');

// Add reminder checkbox element
const createReminderCheckbox = document.getElementById('create_reminder');
const scheduleInput = document.getElementById('scheduled_send_at');

// --- Add Modal DOM Elements ---
const detailsModal = document.getElementById('detailsModal');
const closeModalBtn = document.getElementById('closeModalBtn');
const closeModalBtnBottom = document.getElementById('closeModalBtnBottom');
const modalTitle = document.getElementById('modalTitle');
const modalLoading = document.getElementById('modalLoading');
const modalError = document.getElementById('modalError');
const modalContent = document.getElementById('modalContent');
const modalTableBody = document.getElementById('modalTableBody');

let currentLogs = []; // Store logs in memory

// --- API Functions ---

async function fetchApi(action, method = 'GET', body = null, params = {}) {
    const url = new URL(API_URL, window.location.origin);
    url.searchParams.set('action', action);
    Object.entries(params).forEach(([key, value]) => url.searchParams.set(key, value));

    const options = {
        method: method,
        headers: {},
    };

    if (body) {
        options.body = JSON.stringify(body);
        options.headers['Content-Type'] = 'application/json';
    }
     if (method === 'DELETE') {
         // No body usually for DELETE, ID is in URL param
     }


    try {
        const response = await fetch(url.toString(), options);
        const data = await response.json(); // Attempt to parse JSON regardless of status code

        if (!response.ok) {
             // Use error message from API response if available, otherwise use status text
            const errorMessage = data?.error || response.statusText || `HTTP error ${response.status}`;
            throw new Error(errorMessage);
        }
        return data;
    } catch (error) {
        console.error(`API Error (${action}):`, error);
         // Re-throw for specific handling or return a standard error object
         throw error;
    }
}

// --- Log Rendering ---

function renderLogs() {
    logLoading.classList.add('hidden');
    logError.classList.add('hidden');
    logTableBody.innerHTML = ''; // Clear existing rows

    if (currentLogs.length === 0) {
        logTable.classList.add('hidden');
        noLogs.classList.remove('hidden');
        return;
    }

    noLogs.classList.add('hidden');
    logTable.classList.remove('hidden');

    currentLogs.forEach(log => {
        const row = logTableBody.insertRow();
        row.setAttribute('data-log-id', log.id);

        // --- 1. Title Cell (handles reminders) ---
        const cellTitle = row.insertCell();
        let titleDisplay = `${log.id}. <b>${escapeHtml(log.title)}</b><br>${escapeHtml(log.description)}`;
        if (log.parent_log_id) {
             titleDisplay += `<br><span class="text-xs text-purple-600">(Reminder for #${log.parent_log_id})</span>`;
        }
        cellTitle.innerHTML = titleDisplay;
        //cellTitle.classList.add("whitespace-nowrap"); // Apply styling if needed

        // --- 2. Description Cell ---
        //const cellDesc = row.insertCell();
        //cellDesc.textContent = escapeHtml(log.description);
        //cellDesc.classList.add("whitespace-normal", "break-words", "max-w-xs"); // Apply styling

        // --- 3. Target Cell ---
        const cellTarget = row.insertCell();
        const targetText = log.target_type === 'all' ? 'All Users' :
                           log.target_type === 'specific' ? `<span title="${escapeHtml(log.target_emails || '')}">Specific Users</span>` :
                           log.target_type === 'quiz_inactive_recent' ? 'Quiz Reminder Targets' :
                           log.target_type === 'emw_nincs_jegye' ? 'ÉMW Nincs jegye' :
                           escapeHtml(log.target_type); // Fallback
        cellTarget.innerHTML = targetText;

        // --- 4. Queued At Cell (Previously Sent At) ---
        const cellQueuedAt = row.insertCell();
        cellQueuedAt.textContent = log.sent_at ? new Date(log.sent_at).toLocaleString() : 'N/A';

        // --- 5. Scheduled At Cell (NEW) ---
        const cellScheduledAt = row.insertCell();
        if (log.scheduled_send_at) {
            // Format the scheduled date/time
            cellScheduledAt.textContent = new Date(log.scheduled_send_at).toLocaleString();
        } else {
            // Display something indicating it was immediate or not scheduled
            cellScheduledAt.innerHTML = `<span class="text-gray-400 italic">Immediate</span>`; // Or just leave blank: ''
        }

        // --- 6. Status Cell ---
        const cellStatus = row.insertCell();
        const status = log.fcm_response_status || 'N/A';
        let statusClass = 'text-gray-600'; // Default
        switch (status.toLowerCase()) {
            case 'success': statusClass = 'text-green-600'; break;
            case 'partial_failure': statusClass = 'text-yellow-600'; break;
            case 'failure': statusClass = 'text-red-600'; break;
            case 'processing': statusClass = 'text-blue-600 animate-pulse'; break;
            case 'queued': statusClass = 'text-gray-500'; break;
            case 'no_targets': statusClass = 'text-orange-500'; break; // Example for no targets
        }
        cellStatus.innerHTML = `<span class="${statusClass} font-semibold">${escapeHtml(status)}</span>`;

        // --- 7. Counts Cell ---
        const cellCounts = row.insertCell();
        cellCounts.textContent = `${log.success_count ?? '?'}/${log.failure_count ?? '?'}`;

        // --- 8. Actions Cell ---
        const cellActions = row.insertCell();
        cellActions.classList.add("whitespace-nowrap"); // Prevent wrapping
        let actionsHTML = `
            <button class="copy-btn text-blue-500 hover:text-blue-700 mr-2 text-xs" title="Copy to form">Copy</button>
            <button class="delete-btn text-red-500 hover:red-700 mr-2 text-xs" title="Delete log entry">Delete</button>
            <button class="details-btn text-gray-600 hover:text-gray-800 text-xs" title="View Details">Details</button>
        `;
        // Conditionally add Resume/Retry button if needed (based on your specific logic)
        // if (log.fcm_response_status === 'failed' || log.fcm_response_status === 'processing') {
        //     actionsHTML += `<button class="resume-btn ...">Retry/Resume</button>`;
        // }
        cellActions.innerHTML = actionsHTML;       
    });
}

function escapeHtml(unsafe) {
    if (!unsafe) return '';
    return unsafe
         .replace(/&/g, "&")
         .replace(/</g, "<")
         .replace(/>/g, ">")
         .replace(/"/g, '"')
         .replace(/'/g, "'");
}
// --- Modal Functions ---
function openModal() {
    detailsModal.classList.remove('hidden');
    // Optional: Prevent body scroll when modal is open
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    detailsModal.classList.add('hidden');
    modalError.classList.add('hidden');
    modalContent.classList.add('hidden');
    modalLoading.classList.remove('hidden'); // Reset loading state
    modalTableBody.innerHTML = ''; // Clear content
    document.body.style.overflow = 'auto'; // Restore body scroll
}

async function showDetails(logId) {
    const log = currentLogs.find(l => l.id === logId);
    if (!log) return;

    modalTitle.textContent = `Delivery Details for "${escapeHtml(log.title)}" (Log #${logId})`;
    openModal();

    try {
        const details = await fetchApi('get_delivery_details', 'GET', null, { log_id: logId });

        modalLoading.classList.add('hidden');
        modalTableBody.innerHTML = ''; // Clear previous

        if (details.length === 0) {
             modalTableBody.innerHTML = '<tr><td colspan="4" class="text-center p-4 text-gray-500">No delivery details found for this log entry.</td></tr>';
        } else {
            details.forEach(detail => {
                const row = modalTableBody.insertRow();
                const statusClass = detail.status === 'success' ? 'text-green-600' : 'text-red-600';
                const tokenStart = detail.target_token ? escapeHtml(detail.target_token.substring(0, 15)) + '...' : 'N/A';

                row.innerHTML = `
                    <td class="p-2 border-b">${escapeHtml(detail.target_email || 'N/A')}</td>
                    <td class="p-2 border-b font-mono text-xs" title="${escapeHtml(detail.target_token)}">${tokenStart}</td>
                    <td class="p-2 border-b"><span class="${statusClass} font-semibold">${escapeHtml(detail.status)}</span></td>
                    <td class="p-2 border-b whitespace-normal break-words max-w-xs">${escapeHtml(detail.fcm_error_message || '')}</td>
                `;
            });
        }
        modalContent.classList.remove('hidden');

    } catch (error) {
        modalLoading.classList.add('hidden');
        modalError.textContent = `Error loading details: ${error.message}`;
        modalError.classList.remove('hidden');
    }
}

// --- Event Handlers ---

function handleTargetTypeChange() {
    const selectedType = document.querySelector('input[name="targetType"]:checked').value;
    if (selectedType === 'specific') {
        targetEmailsGroup.style.display = 'block';
        targetEmailsInput.required = true;
    } else {
        targetEmailsGroup.style.display = 'none';
        targetEmailsInput.required = false;
        targetEmailsInput.value = ''; // Clear emails when switching back to 'all'
    }
}

function clearForm() {
    notificationForm.reset();
    targetEmailsGroup.style.display = 'none';
    targetEmailsInput.required = false;
    formStatus.textContent = '';
    formStatus.className = 'mt-4 text-sm';
    sendButton.disabled = false;
    sendButton.textContent = 'Queue Notification';
    document.querySelector('input[name="targetType"][value="all"]').checked = true;
    scheduleInput.value = ''; // Clear schedule input
    createReminderCheckbox.checked = false; // Uncheck reminder box
}
// --- Event Handlers ---
async function handleFormSubmit(event) {
    event.preventDefault();
    // ... (Set loading state) ...

    const formData = new FormData(notificationForm);
    const createReminderFlag = createReminderCheckbox.checked;
    const scheduleLocalValue = formData.get('scheduled_send_at_local');

    // --- Client-side validation: Reminder needs future schedule time ---
    let scheduleTimeIsValidForReminder = false;
    let formattedScheduleTime = null;
    if (scheduleLocalValue) {
         try {
            const date = new Date(scheduleLocalValue);
            if (isNaN(date.getTime())) throw new Error("Invalid date selected.");
            const now = new Date();
            if (date > new Date(now.getTime() + 60000)) { // At least 1 min in future
                 scheduleTimeIsValidForReminder = true;
                  const year = date.getFullYear();
                  const month = String(date.getMonth() + 1).padStart(2, '0');
                  const day = String(date.getDate()).padStart(2, '0');
                  const hours = String(date.getHours()).padStart(2, '0');
                  const minutes = String(date.getMinutes()).padStart(2, '0');
                  formattedScheduleTime = `${year}-${month}-${day} ${hours}:${minutes}:00`;
             }
         } catch (e) { /* handle invalid date format below */ }
    }

    if (createReminderFlag && !scheduleTimeIsValidForReminder) {
        formStatus.textContent = 'Error: To schedule a reminder, please set a Schedule Send Time at least a few minutes in the future.';
        formStatus.className = 'mt-4 text-sm text-red-600';
        sendButton.disabled = false;
        sendButton.textContent = 'Queue Notification';
        scheduleInput.focus();
        return;
    }

    // --- Prepare data for API ---
    const data = {
        title: formData.get('title'),
        description: formData.get('description'),
        targetType: formData.get('targetType'),
        targetEmails: formData.get('targetEmails') || null,
        scheduled_send_at: formattedScheduleTime,
        create_reminder: createReminderFlag
    };

    // --- Validation (Title, Desc, Emails - if needed) ---
    // ...

    formStatus.textContent = data.scheduled_send_at ? 'Scheduling job...' : 'Queueing job...';
    sendButton.textContent = data.scheduled_send_at ? 'Scheduling...' : 'Queueing...';

    try {
        const result = await fetchApi('send', 'POST', data);

        formStatus.textContent = result.message;
        formStatus.className = 'mt-4 text-sm text-green-600';
        clearForm();
        loadLogs();

        // --- Start Polling (CORRECTED LOGIC) ---
        const scheduleDate = result.original_scheduled_send_at
            ? new Date(result.original_scheduled_send_at.replace(' ', 'T') + 'Z') // Adjust timezone handling as needed
            : null;
        // Define pollThreshold *before* using it
        const pollThreshold = new Date(Date.now() + 60 * 60 * 1000); // 1 hour from now

        // Start polling for original job if appropriate
        if (!scheduleDate || scheduleDate <= pollThreshold) {
             if (result.originalLogId && result.status === 'queued') { // Check status from API response
                startPollingLogStatus(result.originalLogId);
             } else if (result.originalLogId && !result.status) {
                 // Fallback if status isn't returned - maybe just poll? Or log warning?
                 console.warn("API response didn't include status, starting polling anyway for original job:", result.originalLogId);
                 startPollingLogStatus(result.originalLogId);
             }
        } else {
             console.log(`Original job ${result.originalLogId} scheduled for later, not polling immediately.`);
        }

         // Start polling for the reminder job if appropriate
         if (result.reminderLogId && result.reminder_scheduled_send_at) {
              const reminderScheduleDate = new Date(result.reminder_scheduled_send_at.replace(' ', 'T') + 'Z'); // Adjust timezone
              if (reminderScheduleDate <= pollThreshold) { // Use same threshold
                   // We assume reminder is always 'queued' initially, so no status check needed here
                   startPollingLogStatus(result.reminderLogId);
              } else {
                  console.log(`Reminder job ${result.reminderLogId} scheduled for later, not polling immediately.`);
              }
         }

    } catch (error) {
        formStatus.textContent = `Error queueing job: ${error.message}`;
        formStatus.className = 'mt-4 text-sm text-red-600';
        // Keep button disabled until form cleared or user edits? Or re-enable?
        // sendButton.disabled = false;
        // sendButton.textContent = 'Queue Notification';
    } finally {
        // Button state managed by clearForm or error handling
    }
}

// Polling mechanism
const pollingIntervals = {}; // Store interval IDs: { logId: intervalId }

function startPollingLogStatus(logId) {
    if (pollingIntervals[logId]) {
        clearInterval(pollingIntervals[logId]); // Clear existing interval if any
    }
    console.log(`Starting polling for logId: ${logId}`);

    // Add placeholder status to the log table row immediately if possible
    updateLogStatusInTable(logId, 'queued', 0, 0); // Show initial queued status

    pollingIntervals[logId] = setInterval(async () => {
        try {
            const statusData = await fetchApi('get_log_status', 'GET', null, { log_id: logId });
            console.log(`Polling logId ${logId}, status: ${statusData.fcm_response_status}`);

            // Update status in the main log table
            updateLogStatusInTable(logId, statusData.fcm_response_status, statusData.success_count, statusData.failure_count);

            // Check for final states
            const finalStates = ['success', 'partial_failure', 'failure', 'no_targets']; // Add any other final states
            if (finalStates.includes(statusData.fcm_response_status)) {
                console.log(`Polling finished for logId: ${logId}, final status: ${statusData.fcm_response_status}`);
                clearInterval(pollingIntervals[logId]);
                delete pollingIntervals[logId];
                 // Maybe add a final notification to the user in formStatus area
                  formStatus.textContent = `Job #${logId} finished with status: ${statusData.fcm_response_status}`;
                  formStatus.className = 'mt-4 text-sm text-gray-700';
            }
        } catch (error) {
            console.error(`Error polling status for logId ${logId}:`, error);
            // Decide if polling should stop on error
             clearInterval(pollingIntervals[logId]);
             delete pollingIntervals[logId];
        }
    }, 5000); // Poll every 5 seconds (adjust frequency as needed)
}

// Helper to update the status directly in the log table row
function updateLogStatusInTable(logId, status, successCount, failureCount) {
    const row = logTableBody.querySelector(`tr[data-log-id="${logId}"]`);
    if (row) {
        const statusCell = row.cells[4]; // Assuming status is the 5th cell (index 4)
        const countsCell = row.cells[5]; // Assuming counts is the 6th cell (index 5)

        if (statusCell) {
             const statusClass = status === 'success' ? 'text-green-600' :
                                 status === 'partial_failure' ? 'text-yellow-600' :
                                 status === 'processing' ? 'text-blue-600 animate-pulse' : // Indicate processing
                                 status === 'queued' ? 'text-gray-500' :
                                 'text-red-600'; // failure or other
            statusCell.innerHTML = `<span class="${statusClass} font-semibold">${escapeHtml(status || 'N/A')}</span>`;
        }
         if (countsCell) {
             countsCell.textContent = `${successCount ?? '?'}/${failureCount ?? '?'}`;
         }

        // Handle Resume button visibility based on new status (if you implement retry)
        const resumeBtn = row.querySelector('.resume-btn');
        if (resumeBtn) {
            // Remove resume button if status is no longer 'processing' or relevant state
            if (status !== 'failed') { // Example: only show retry for 'failed' state
                resumeBtn.remove();
            } else {
                resumeBtn.textContent = 'Retry'; // Change text if needed
                 resumeBtn.disabled = false; // Re-enable if previously disabled
            }
        }

    }
}

function handleCopyClick(event) {
    
    if (!event.target.classList.contains('copy-btn')) return;

    const row = event.target.closest('tr');
    const logId = parseInt(row.getAttribute('data-log-id'), 10);
    const log = currentLogs.find(l => l.id === logId);

    if (log) {
        titleInput.value = log.title;
        descriptionInput.value = log.description;
        // Don't automatically set target type or emails, as user might want to change them
        // Maybe scroll to form?
        notificationForm.scrollIntoView({ behavior: 'smooth' });
        titleInput.focus();
        formStatus.textContent = `Copied content from log #${logId}. Please review target settings before sending.`;
        formStatus.className = 'mt-4 text-sm text-blue-600';
    }
}

async function handleDeleteClick(event) {
    if (!event.target.classList.contains('delete-btn')) return;

    const row = event.target.closest('tr');
    const logId = parseInt(row.getAttribute('data-log-id'), 10);
    const log = currentLogs.find(l => l.id === logId);

    if (log && confirm(`Are you sure you want to delete the log for "${log.title}"?`)) {
        // Stop polling this log if it's being deleted
         if (pollingIntervals[logId]) {
              clearInterval(pollingIntervals[logId]);
              delete pollingIntervals[logId];
              console.log(`Stopped polling for deleted logId: ${logId}`);
         }
        try {
            event.target.textContent = 'Deleting...';
            event.target.disabled = true;
            await fetchApi('delete_log', 'DELETE', null, { id: logId });
            // Remove from UI immediately
            row.remove();
            // Update internal state
            currentLogs = currentLogs.filter(l => l.id !== logId);
            // Re-check if logs are now empty
             if (currentLogs.length === 0) {
                logTable.classList.add('hidden');
                noLogs.classList.remove('hidden');
            }

            formStatus.textContent = `Log entry #${logId} deleted.`;
            formStatus.className = 'mt-4 text-sm text-green-600';

        } catch (error) {
             formStatus.textContent = `Error deleting log #${logId}: ${error.message}`;
             formStatus.className = 'mt-4 text-sm text-red-600';
             event.target.textContent = 'Delete'; // Reset button text on error
             event.target.disabled = false; // Re-enable button on error
        }
    }
}

function handleDetailsClick(event) {
    if (!event.target.classList.contains('details-btn')) return;

    const row = event.target.closest('tr');
    const logId = parseInt(row.getAttribute('data-log-id'), 10);
    showDetails(logId);
}
// New handler for Resume button clicks (using event delegation)
async function handleResumeClick(event) {
    if (!event.target.classList.contains('resume-btn')) return;

    const row = event.target.closest('tr');
    const logId = parseInt(row.getAttribute('data-log-id'), 10);
    const log = currentLogs.find(l => l.id === logId);

    if (log && confirm(`Are you sure you want to resume the sending process for "${log.title}" (Log #${logId})?`)) {
        const resumeButton = event.target;
        resumeButton.disabled = true;
        resumeButton.textContent = 'Resuming...';
        formStatus.textContent = `Resuming job #${logId}...`;
        formStatus.className = 'mt-4 text-sm text-blue-600';

        try {
            const result = await fetchApi('resume_send', 'POST', { log_id: logId });
            formStatus.textContent = `Resume Complete: ${result.message} (Status: ${result.status}, Total Success: ${result.totalSuccessCount}, Total Fail: ${result.totalFailureCount})`;
            formStatus.className = 'mt-4 text-sm text-green-600';
            // Refresh logs to update the status and remove/update the button
            loadLogs();
        } catch (error) {
            formStatus.textContent = `Error resuming job #${logId}: ${error.message}`;
            formStatus.className = 'mt-4 text-sm text-red-600';
            resumeButton.disabled = false; // Re-enable on error
            resumeButton.textContent = 'Resume';
        }
    }
}

// --- Initialization ---

async function loadLogs() {
    logLoading.classList.remove('hidden');
    logError.classList.add('hidden');
    logTable.classList.add('hidden');
    noLogs.classList.add('hidden');

    try {
        currentLogs = await fetchApi('get_logs');
        renderLogs();
    } catch (error) {
        logLoading.classList.add('hidden');
        logError.textContent = `Failed to load logs: ${error.message}`;
        logError.classList.remove('hidden');
    }
}

// --- Attach Event Listeners ---
notificationForm.addEventListener('submit', handleFormSubmit);
clearFormButton.addEventListener('click', clearForm);
targetTypeRadios.forEach(radio => radio.addEventListener('change', handleTargetTypeChange));
// Attach listeners to table body for delegation
logTableBody.addEventListener('click', handleCopyClick);
logTableBody.addEventListener('click', handleDeleteClick);
logTableBody.addEventListener('click', handleDetailsClick);
logTableBody.addEventListener('click', handleResumeClick); // Add listener for resume clicks
// Modal close listeners
closeModalBtn.addEventListener('click', closeModal);
closeModalBtnBottom.addEventListener('click', closeModal);
// Optional: Close modal if clicking outside the content area
detailsModal.addEventListener('click', (event) => {
    if (event.target === detailsModal) { // Check if click is on the background overlay itself
        closeModal();
    }
});

// --- Initial Load ---
document.addEventListener('DOMContentLoaded', () => {
    loadLogs();
    handleTargetTypeChange(); // Set initial state for email input visibility
});