/* 
Design a mysql database for a SPA, that sends push notifications to users from 'inspiracio_fbuser' table (its has columns like: id, name, mail, token). 
I have to specify by the mail column to which user I want to send the push notification. Or I can send it to all users.
I have to log this push notifications in 'push_notification_logs' table.
The title maximum length is 47 characters.
The description maximum length is 400 characters.

Then build a SPA on this database with html5, tailwindcss, javascript (es6), and php7.3.
The api.php file should be in one file.
The app should be capable to send push notifications to specified users. 
The app should be capable to send push notifications to all users.
The app should be capable to copy a sent notification to a new notification.
The app should be capable to delete a not sent notification.

For the push notifications use firebase cloud messaging wuth this url: https://fcm.googleapis.com/v1/projects/homeinfo-vip/messages:send

CREATE TABLE `push_notification_logs` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `title` varchar(47) NOT NULL,
  `description` varchar(400) NOT NULL,
  `target_type` enum('all','specific') NOT NULL COMMENT 'Indicates if sent to all users or specific emails',
  `target_emails` text DEFAULT NULL COMMENT 'Comma-separated list of emails if target_type is specific',
  `sent_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `fcm_response_status` varchar(100) DEFAULT NULL COMMENT 'Overall status from FCM attempt (e.g., success, partial_failure, failure)',
  `fcm_response_details` text DEFAULT NULL COMMENT 'Detailed response/errors from FCM, potentially JSON',
  `success_count` int(11) DEFAULT 0 COMMENT 'Number of successful deliveries reported by FCM',
  `failure_count` int(11) DEFAULT 0 COMMENT 'Number of failed deliveries reported by FCM',
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4

-- New table: 'push_notification_delivery_status'
-- Stores the outcome for each individual token targeted in a push notification log entry.
CREATE TABLE `push_notification_delivery_status` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `push_notification_log_id` int(11) NOT NULL COMMENT 'Foreign key linking to the main log entry',
  `target_user_id` int(11) DEFAULT NULL COMMENT 'FK to inspiracio_fbuser.id, if available/fetched',
  `target_email` varchar(255) DEFAULT NULL COMMENT 'The email associated with the token (for reference)',
  `target_token` varchar(255) NOT NULL COMMENT 'The specific FCM token that was targeted',
  `status` enum('success','failure') NOT NULL COMMENT 'Delivery status for this specific token',
  `fcm_error_message` text DEFAULT NULL COMMENT 'Specific error message from FCM if status is failure',
  `attempted_at` timestamp NOT NULL DEFAULT current_timestamp() COMMENT 'Timestamp of this specific attempt',
  PRIMARY KEY (`id`),
  KEY `idx_log_id` (`push_notification_log_id`), -- Essential for retrieving details for a log
  KEY `idx_target_token` (`target_token`),     -- May be useful for debugging specific tokens
  -- Foreign Key constraint for data integrity: ensures log_id exists and cascades deletes
  CONSTRAINT `fk_delivery_log` FOREIGN KEY (`push_notification_log_id`) REFERENCES `push_notification_logs` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
  -- Optional: FK to user table if user_id is consistently populated
  -- CONSTRAINT `fk_delivery_user` FOREIGN KEY (`target_user_id`) REFERENCES `inspiracio_fbuser` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

*/

CREATE TABLE push_notifications (
    id INT(6) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(47) NOT NULL, -- optimal 47 characters
    description TEXT NOT NULL,  -- optimal is 50 characters, but up to 400 characters is ok
    image TEXT DEFAULT NULL,
    url TEXT DEFAULT NULL,
    status enum('created', 'in_progress', 'done') NOT NULL DEFAULT 'created',    
    updated_at TIMESTAMP DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP    
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_hungarian_ci;

CREATE TABLE push_notification_to_users (
    id INT(6) UNSIGNED AUTO_INCREMENT PRIMARY KEY,    
    fbuser_id INT(6) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (fbuser_id) REFERENCES inspiracio_fbuser(id) ON DELETE CASCADE
);
CREATE TABLE push_notification_logs (
    id INT(6) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    push_notification_id INT(6) NOT NULL,    
    token TEXT NOT NULL,
    status_code INT(3) UNSIGNED NOT NULL,
    respone JSON NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_hungarian_ci;


-- Modify the push_notification_logs table
ALTER TABLE `push_notification_logs`
MODIFY COLUMN `fcm_response_status` enum(
    'queued',          -- Newly added: Job is waiting to be processed
    'processing',      -- Newly added: Worker has picked up the job
    'success',
    'partial_failure',
    'failure',
    'no_targets'       -- Optional: Keep if you have specific handling for 0 targets
    -- Add 'cancelled' or 'error' if needed
) DEFAULT NULL COMMENT 'Overall status of the push notification job';

-- Optional but Recommended: Add an index for the worker to find jobs
ALTER TABLE `push_notification_logs`
ADD INDEX `idx_status_sent_at` (`fcm_response_status`, `sent_at`);

-- Add a column to store the scheduled send time
ALTER TABLE `push_notification_logs`
ADD COLUMN `scheduled_send_at` DATETIME NULL DEFAULT NULL COMMENT 'Time the push notification should be sent' AFTER `sent_at`;
-- sent_at now represents the time it was *queued* or initially created

-- Optional: Add an index to help the worker find scheduled jobs
ALTER TABLE `push_notification_logs`
ADD INDEX `idx_status_scheduled_send_at` (`fcm_response_status`, `scheduled_send_at`);

-- Add parent link and modify target_type enum if needed
ALTER TABLE `push_notification_logs`
ADD COLUMN `parent_log_id` INT(11) NULL DEFAULT NULL COMMENT 'FK to the original log entry if this is a reminder' AFTER `id`,
ADD CONSTRAINT `fk_parent_log` FOREIGN KEY (`parent_log_id`) REFERENCES `push_notification_logs` (`id`) ON DELETE SET NULL ON UPDATE CASCADE;

-- Optional: Add index for parent_log_id
ALTER TABLE `push_notification_logs`
ADD INDEX `idx_parent_log_id` (`parent_log_id`);

-- Ensure target_type enum includes the quiz reminder type
ALTER TABLE `push_notification_logs`
MODIFY COLUMN `target_type` enum(
    'all',
    'specific',
    'quiz_inactive_recent' -- Make sure this exists
) DEFAULT NULL COMMENT 'Indicates the target audience determination method';

-- Note: Also ensure the fcm_response_status enum includes 'queued', 'processing', etc. from previous steps
-- Example:
-- ALTER TABLE `push_notification_logs` MODIFY COLUMN `fcm_response_status`
-- enum('queued','processing','success','partial_failure','failure','no_targets')
-- DEFAULT NULL COMMENT 'Overall status of the push notification job';